@extends('layouts.user')

@section('title', 'ثبت نظر جدید')

@section('styles')
<style>
    /* Modern Rating Create Page Styles */
    .rating-create-container {
        background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
        min-height: 100vh;
        padding: 2rem 0;
    }

    .page-header {
        background: white;
        border-radius: 20px;
        padding: 2rem;
        margin-bottom: 2rem;
        box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
        position: relative;
        overflow: hidden;
    }

    .page-header::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 100px;
        background: linear-gradient(135deg, #667eea, #764ba2);
        z-index: 1;
    }

    .page-header-content {
        position: relative;
        z-index: 2;
        color: white;
        margin-top: 20px;
    }

    .page-title {
        font-size: 2.5rem;
        font-weight: 700;
        margin-bottom: 0.5rem;
    }

    .page-subtitle {
        font-size: 1.1rem;
        opacity: 0.9;
        margin-bottom: 2rem;
    }

    .user-stats {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
        gap: 1rem;
        margin-top: 2rem;
    }

    .user-stat {
        background: white;
        border-radius: 12px;
        padding: 1rem;
        text-align: center;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    }

    .user-stat-number {
        font-size: 1.5rem;
        font-weight: 700;
        color: #667eea;
        display: block;
    }

    .user-stat-label {
        font-size: 0.75rem;
        color: #718096;
        margin-top: 0.25rem;
    }

    .form-container {
        background: white;
        border-radius: 20px;
        padding: 2rem;
        margin-bottom: 2rem;
        box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
    }

    .form-section {
        margin-bottom: 2rem;
        padding-bottom: 2rem;
        border-bottom: 2px solid #f1f5f9;
    }

    .form-section:last-child {
        border-bottom: none;
        margin-bottom: 0;
        padding-bottom: 0;
    }

    .section-title {
        font-size: 1.25rem;
        font-weight: 700;
        color: #2d3748;
        margin-bottom: 1.5rem;
        display: flex;
        align-items: center;
        gap: 0.75rem;
    }

    .section-icon {
        width: 40px;
        height: 40px;
        background: linear-gradient(135deg, #667eea, #764ba2);
        color: white;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.1rem;
    }

    .modern-form-group {
        margin-bottom: 1.5rem;
    }

    .modern-label {
        display: block;
        font-weight: 600;
        color: #374151;
        margin-bottom: 0.5rem;
        font-size: 0.875rem;
    }

    .required-label::after {
        content: ' *';
        color: #ef4444;
    }

    .modern-input {
        width: 100%;
        padding: 0.875rem 1rem;
        border: 2px solid #e5e7eb;
        border-radius: 12px;
        font-size: 1rem;
        transition: all 0.3s ease;
        background: #f9fafb;
    }

    .modern-input:focus {
        outline: none;
        border-color: #667eea;
        background: white;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }

    .modern-select {
        appearance: none;
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
        background-position: left 0.75rem center;
        background-repeat: no-repeat;
        background-size: 1.5em 1.5em;
        padding-left: 3rem;
    }

    .modern-textarea {
        resize: vertical;
        min-height: 120px;
    }

    .location-selector {
        position: relative;
    }

    .location-search {
        position: relative;
    }

    .location-search-input {
        padding-left: 3rem;
    }

    .location-search-icon {
        position: absolute;
        left: 1rem;
        top: 50%;
        transform: translateY(-50%);
        color: #9ca3af;
        font-size: 1.25rem;
    }

    .location-dropdown {
        position: absolute;
        top: 100%;
        left: 0;
        right: 0;
        background: white;
        border: 2px solid #e5e7eb;
        border-top: none;
        border-radius: 0 0 12px 12px;
        max-height: 300px;
        overflow-y: auto;
        z-index: 1000;
        display: none;
    }

    .location-dropdown.show {
        display: block;
    }

    .location-option {
        padding: 1rem;
        cursor: pointer;
        border-bottom: 1px solid #f1f5f9;
        transition: all 0.2s ease;
    }

    .location-option:hover {
        background: #f8fafc;
    }

    .location-option:last-child {
        border-bottom: none;
    }

    .location-name {
        font-weight: 600;
        color: #2d3748;
        margin-bottom: 0.25rem;
    }

    .location-meta {
        font-size: 0.875rem;
        color: #718096;
    }

    .selected-location {
        background: #f0f9ff;
        border: 2px solid #0ea5e9;
        border-radius: 12px;
        padding: 1rem;
        margin-bottom: 1rem;
    }

    .selected-location-name {
        font-weight: 700;
        color: #0c4a6e;
        margin-bottom: 0.5rem;
    }

    .selected-location-meta {
        font-size: 0.875rem;
        color: #0369a1;
    }

    .change-location-btn {
        background: #0ea5e9;
        color: white;
        border: none;
        padding: 0.5rem 1rem;
        border-radius: 8px;
        font-size: 0.875rem;
        cursor: pointer;
        margin-top: 0.5rem;
        transition: all 0.3s ease;
    }

    .change-location-btn:hover {
        background: #0284c7;
    }

    .rating-stars-container {
        display: flex;
        align-items: center;
        gap: 1rem;
        flex-wrap: wrap;
    }

    .rating-stars {
        display: flex;
        gap: 0.5rem;
    }

    .star-input {
        display: none;
    }

    .star-label {
        cursor: pointer;
        font-size: 2rem;
        color: #e5e7eb;
        transition: all 0.3s ease;
        position: relative;
    }

    /* Remove CSS hover effects - handled by JavaScript */
    .star-label i {
        transition: all 0.3s ease;
    }

    .rating-value {
        font-size: 1.25rem;
        font-weight: 700;
        color: #667eea;
        min-width: 80px;
    }

    .rating-text {
        font-size: 0.875rem;
        color: #718096;
        margin-top: 0.25rem;
    }

    .detailed-ratings {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1.5rem;
    }

    .detailed-rating {
        background: #f8fafc;
        border-radius: 12px;
        padding: 1rem;
        border: 1px solid #e2e8f0;
    }

    .detailed-rating-label {
        font-weight: 600;
        color: #374151;
        margin-bottom: 0.75rem;
        font-size: 0.875rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .detailed-rating-stars {
        display: flex;
        gap: 0.25rem;
    }

    .detailed-star-label {
        cursor: pointer;
        font-size: 1.25rem;
        color: #e5e7eb;
        transition: all 0.2s ease;
    }

    /* Remove CSS hover effects for detailed stars - handled by JavaScript */
    .detailed-star-label i {
        transition: all 0.2s ease;
    }

    .comments-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 2rem;
    }

    .comment-section {
        position: relative;
    }

    .comment-icon {
        position: absolute;
        top: 2.5rem;
        left: 1rem;
        color: #9ca3af;
        font-size: 1.25rem;
    }

    .comment-textarea {
        padding-left: 3rem;
    }

    .character-count {
        font-size: 0.75rem;
        color: #9ca3af;
        text-align: right;
        margin-top: 0.5rem;
    }

    .trip-details {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1.5rem;
    }

    .suggestions-section {
        background: white;
        border-radius: 16px;
        padding: 1.5rem;
        margin-bottom: 2rem;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .suggestions-title {
        font-size: 1.25rem;
        font-weight: 700;
        color: #2d3748;
        margin-bottom: 1rem;
        display: flex;
        align-items: center;
        gap: 0.75rem;
    }

    .suggestions-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
        gap: 1rem;
    }

    .suggestion-card {
        background: #f8fafc;
        border-radius: 12px;
        padding: 1rem;
        border: 1px solid #e2e8f0;
        cursor: pointer;
        transition: all 0.3s ease;
    }

    .suggestion-card:hover {
        background: #f1f5f9;
        border-color: #667eea;
        transform: translateY(-2px);
    }

    .suggestion-name {
        font-weight: 600;
        color: #2d3748;
        margin-bottom: 0.25rem;
    }

    .suggestion-meta {
        font-size: 0.875rem;
        color: #718096;
    }

    .form-actions {
        display: flex;
        gap: 1rem;
        justify-content: space-between;
        align-items: center;
        margin-top: 2rem;
        padding-top: 2rem;
        border-top: 2px solid #f1f5f9;
    }

    .modern-btn {
        background: linear-gradient(135deg, #667eea, #764ba2);
        color: white;
        border: none;
        padding: 0.875rem 2rem;
        border-radius: 12px;
        font-weight: 600;
        font-size: 1rem;
        cursor: pointer;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        text-decoration: none;
    }

    .modern-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        color: white;
        text-decoration: none;
    }

    .modern-btn-secondary {
        background: #6b7280;
    }

    .modern-btn-secondary:hover {
        background: #4b5563;
        box-shadow: 0 8px 25px rgba(107, 114, 128, 0.3);
    }

    .modern-btn-warning {
        background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
        color: white;
        border: none;
    }

    .modern-btn-warning:hover {
        background: linear-gradient(135deg, #d97706 0%, #b45309 100%);
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(245, 158, 11, 0.3);
    }

    .form-info {
        background: linear-gradient(135deg, #e0f2fe, #b3e5fc);
        border: 1px solid #0ea5e9;
        border-radius: 12px;
        padding: 1rem;
        margin-bottom: 2rem;
        display: flex;
        align-items: center;
        gap: 0.75rem;
    }

    .form-info-icon {
        color: #0369a1;
        font-size: 1.25rem;
    }

    .form-info-text {
        color: #0c4a6e;
        font-size: 0.875rem;
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .rating-create-container {
            padding: 1rem 0;
        }

        .page-header {
            margin: 1rem;
            padding: 1.5rem;
        }

        .page-title {
            font-size: 2rem;
        }

        .user-stats {
            grid-template-columns: repeat(2, 1fr);
        }

        .detailed-ratings {
            grid-template-columns: 1fr;
        }

        .comments-grid {
            grid-template-columns: 1fr;
            gap: 1rem;
        }

        .trip-details {
            grid-template-columns: 1fr;
        }

        .suggestions-grid {
            grid-template-columns: 1fr;
        }

        .form-actions {
            flex-direction: column;
            gap: 1rem;
        }

        .rating-stars-container {
            flex-direction: column;
            align-items: flex-start;
        }
    }

    /* Animation */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    .form-container {
        animation: fadeInUp 0.6s ease forwards;
    }

    .suggestions-section {
        animation: fadeInUp 0.6s ease forwards;
        animation-delay: 0.2s;
    }
</style>
@endsection

@section('content')
<div class="rating-create-container">
    <div class="container-fluid">
        <!-- Page Header -->
        <div class="page-header">
            <div class="page-header-content">
                <h1 class="page-title">ثبت نظر جدید</h1>
                <p class="page-subtitle">تجربه خود را با سایر کاربران به اشتراک بگذارید</p>

                <div class="d-flex justify-content-end">
                    <a href="{{ route('user.ratings.index') }}" class="modern-btn modern-btn-secondary">
                        <i class="fas fa-arrow-right"></i>
                        بازگشت به لیست
                    </a>
                </div>
            </div>

            <!-- User Statistics -->
            <div class="user-stats">
                <div class="user-stat">
                    <span class="user-stat-number">{{ $userStats['total'] }}</span>
                    <span class="user-stat-label">کل نظرات</span>
                </div>

                <div class="user-stat">
                    <span class="user-stat-number">{{ $userStats['this_month'] }}</span>
                    <span class="user-stat-label">این ماه</span>
                </div>

                <div class="user-stat">
                    <span class="user-stat-number">{{ number_format($userStats['avg_rating'], 1) }}</span>
                    <span class="user-stat-label">میانگین امتیاز</span>
                </div>
            </div>
        </div>

        <!-- Suggestions Section -->
        @if($popularLocations->count() > 0 || $recentLocations->count() > 0)
        <div class="suggestions-section">
            <div class="suggestions-title">
                <div class="section-icon">
                    <i class="fas fa-lightbulb"></i>
                </div>
                پیشنهادات برای شما
            </div>

            @if($popularLocations->count() > 0)
            <h6 class="mb-3">مکان‌های محبوب</h6>
            <div class="suggestions-grid mb-4">
                @foreach($popularLocations as $popular)
                <div class="suggestion-card" onclick="selectLocation('{{ $popular->title_id }}', '{{ $popular->title }}', '{{ $popular->categoryRelation ? $popular->categoryRelation->category_fa : '' }}')">
                    <div class="suggestion-name">{{ $popular->title }}</div>
                    <div class="suggestion-meta">
                        {{ $popular->categoryRelation ? $popular->categoryRelation->category_fa : '' }}
                        • {{ $popular->ratings_count }} نظر
                    </div>
                </div>
                @endforeach
            </div>
            @endif

            @if($recentLocations->count() > 0)
            <h6 class="mb-3">مکان‌های جدید</h6>
            <div class="suggestions-grid">
                @foreach($recentLocations as $recent)
                <div class="suggestion-card" onclick="selectLocation('{{ $recent->title_id }}', '{{ $recent->title }}', '{{ $recent->categoryRelation ? $recent->categoryRelation->category_fa : '' }}')">
                    <div class="suggestion-name">{{ $recent->title }}</div>
                    <div class="suggestion-meta">
                        {{ $recent->categoryRelation ? $recent->categoryRelation->category_fa : '' }}
                        • جدید
                    </div>
                </div>
                @endforeach
            </div>
            @endif
        </div>
        @endif

        <!-- Main Form -->
        <div class="form-container">
            <form method="POST" action="{{ route('user.ratings.store') }}" id="ratingForm">
                @csrf

                <!-- Location Selection Section -->
                <div class="form-section">
                    <div class="section-title">
                        <div class="section-icon">
                            <i class="fas fa-map-marker-alt"></i>
                        </div>
                        انتخاب مکان
                    </div>

                    @if(isset($location))
                    <!-- Pre-selected Location -->
                    <div class="selected-location">
                        <div class="selected-location-name">{{ $location->title }}</div>
                        <div class="selected-location-meta">
                            {{ $location->categoryRelation ? $location->categoryRelation->category_fa : '' }}
                            @if($location->province || $location->city)
                            • {{ $location->province ? $location->province->name_fa : '' }}
                            @if($location->city)
                            - {{ $location->city->name_fa }}
                            @endif
                            @endif
                        </div>
                        <button type="button" class="change-location-btn" onclick="changeLocation()">
                            تغییر مکان
                        </button>
                    </div>
                    <input type="hidden" name="title_id" value="{{ $location->title_id }}" id="selectedLocationId">
                    @else
                    <!-- Location Selector -->
                    <div class="modern-form-group">
                        <label class="modern-label required-label">انتخاب مکان</label>
                        <div class="location-selector">
                            <div class="location-search">
                                <input type="text"
                                       class="modern-input location-search-input"
                                       id="locationSearch"
                                       placeholder="جستجو در مکان‌ها..."
                                       autocomplete="off">
                                <i class="fas fa-search location-search-icon"></i>
                                <div class="location-dropdown" id="locationDropdown">
                                    @foreach($locations as $loc)
                                    <div class="location-option"
                                         data-id="{{ $loc->title_id }}"
                                         data-title="{{ $loc->title }}"
                                         data-category="{{ $loc->categoryRelation ? $loc->categoryRelation->category_fa : '' }}">
                                        <div class="location-name">{{ $loc->title }}</div>
                                        <div class="location-meta">
                                            {{ $loc->categoryRelation ? $loc->categoryRelation->category_fa : '' }}
                                            @if($loc->province || $loc->city)
                                            • {{ $loc->province ? $loc->province->name_fa : '' }}
                                            @if($loc->city)
                                            - {{ $loc->city->name_fa }}
                                            @endif
                                            @endif
                                        </div>
                                    </div>
                                    @endforeach
                                </div>
                            </div>
                            <input type="hidden" name="title_id" id="selectedLocationId" required>
                        </div>
                        @error('title_id')
                        <div class="text-danger mt-2">{{ $message }}</div>
                        @enderror
                    </div>
                    @endif
                </div>

                <!-- Overall Rating Section -->
                <div class="form-section">
                    <div class="section-title">
                        <div class="section-icon">
                            <i class="fas fa-star"></i>
                        </div>
                        امتیاز کلی
                    </div>

                    <div class="modern-form-group">
                        <label class="modern-label required-label">امتیاز شما</label>
                        <div class="rating-stars-container">
                            <div class="rating-stars" id="overallRating">
                                @for($i = 1; $i <= 5; $i++)
                                    <input type="radio" name="rating" id="rating-{{ $i }}" value="{{ $i }}" class="star-input" {{ old('rating') == $i ? 'checked' : '' }} required>
                                    <label for="rating-{{ $i }}" class="star-label">
                                        <i class="fas fa-star"></i>
                                    </label>
                                @endfor
                            </div>
                            <div class="rating-value" id="ratingValue">0/5</div>
                        </div>
                        <div class="rating-text" id="ratingText">امتیاز خود را انتخاب کنید</div>
                        @error('rating')
                        <div class="text-danger mt-2">{{ $message }}</div>
                        @enderror
                    </div>
                </div>

                <!-- Detailed Ratings Section -->
                <div class="form-section">
                    <div class="section-title">
                        <div class="section-icon">
                            <i class="fas fa-chart-bar"></i>
                        </div>
                        امتیازات تفصیلی
                    </div>

                    <div class="detailed-ratings">
                        <!-- Service Rating -->
                        <div class="detailed-rating">
                            <div class="detailed-rating-label">
                                <i class="fas fa-concierge-bell"></i>
                                کیفیت خدمات
                            </div>
                            <div class="detailed-rating-stars">
                                @for($i = 1; $i <= 5; $i++)
                                    <input type="radio" name="service_rating" id="service-{{ $i }}" value="{{ $i }}" class="detailed-star-input" {{ old('service_rating') == $i ? 'checked' : '' }}>
                                    <label for="service-{{ $i }}" class="detailed-star-label">
                                        <i class="fas fa-star"></i>
                                    </label>
                                @endfor
                            </div>
                        </div>

                        <!-- Cleanliness Rating -->
                        <div class="detailed-rating">
                            <div class="detailed-rating-label">
                                <i class="fas fa-broom"></i>
                                نظافت
                            </div>
                            <div class="detailed-rating-stars">
                                @for($i = 1; $i <= 5; $i++)
                                    <input type="radio" name="cleanliness_rating" id="cleanliness-{{ $i }}" value="{{ $i }}" class="detailed-star-input" {{ old('cleanliness_rating') == $i ? 'checked' : '' }}>
                                    <label for="cleanliness-{{ $i }}" class="detailed-star-label">
                                        <i class="fas fa-star"></i>
                                    </label>
                                @endfor
                            </div>
                        </div>

                        <!-- Location Rating -->
                        <div class="detailed-rating">
                            <div class="detailed-rating-label">
                                <i class="fas fa-map-marker-alt"></i>
                                موقعیت
                            </div>
                            <div class="detailed-rating-stars">
                                @for($i = 1; $i <= 5; $i++)
                                    <input type="radio" name="location_rating" id="location-{{ $i }}" value="{{ $i }}" class="detailed-star-input" {{ old('location_rating') == $i ? 'checked' : '' }}>
                                    <label for="location-{{ $i }}" class="detailed-star-label">
                                        <i class="fas fa-star"></i>
                                    </label>
                                @endfor
                            </div>
                        </div>

                        <!-- Price Rating -->
                        <div class="detailed-rating">
                            <div class="detailed-rating-label">
                                <i class="fas fa-dollar-sign"></i>
                                ارزش نسبت به قیمت
                            </div>
                            <div class="detailed-rating-stars">
                                @for($i = 1; $i <= 5; $i++)
                                    <input type="radio" name="price_rating" id="price-{{ $i }}" value="{{ $i }}" class="detailed-star-input" {{ old('price_rating') == $i ? 'checked' : '' }}>
                                    <label for="price-{{ $i }}" class="detailed-star-label">
                                        <i class="fas fa-star"></i>
                                    </label>
                                @endfor
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Comments Section -->
                <div class="form-section">
                    <div class="section-title">
                        <div class="section-icon">
                            <i class="fas fa-comment"></i>
                        </div>
                        نظرات و تجربیات
                    </div>

                    <div class="comments-grid">
                        <!-- Positive Comments -->
                        <div class="comment-section">
                            <div class="modern-form-group">
                                <label class="modern-label">نکات مثبت</label>
                                <i class="fas fa-thumbs-up comment-icon"></i>
                                <textarea class="modern-input modern-textarea comment-textarea"
                                          id="positive_comment"
                                          name="positive_comment"
                                          placeholder="چه چیزهایی را دوست داشتید؟"
                                          maxlength="500">{{ old('positive_comment') }}</textarea>
                                <div class="character-count">
                                    <span id="positiveCount">0</span>/500 کاراکتر
                                </div>
                                @error('positive_comment')
                                <div class="text-danger mt-2">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <!-- Negative Comments -->
                        <div class="comment-section">
                            <div class="modern-form-group">
                                <label class="modern-label">نکات منفی</label>
                                <i class="fas fa-thumbs-down comment-icon"></i>
                                <textarea class="modern-input modern-textarea comment-textarea"
                                          id="negative_comment"
                                          name="negative_comment"
                                          placeholder="چه چیزهایی قابل بهبود است؟"
                                          maxlength="500">{{ old('negative_comment') }}</textarea>
                                <div class="character-count">
                                    <span id="negativeCount">0</span>/500 کاراکتر
                                </div>
                                @error('negative_comment')
                                <div class="text-danger mt-2">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Trip Details Section -->
                <div class="form-section">
                    <div class="section-title">
                        <div class="section-icon">
                            <i class="fas fa-suitcase"></i>
                        </div>
                        جزئیات سفر
                    </div>

                    <div class="trip-details">
                        <!-- Trip Type -->
                        <div class="modern-form-group">
                            <label class="modern-label">نوع سفر</label>
                            <select class="modern-input modern-select" id="trip_type" name="trip_type">
                                <option value="">انتخاب کنید</option>
                                <option value="تنها" {{ old('trip_type') == 'تنها' ? 'selected' : '' }}>تنها</option>
                                <option value="زوجی" {{ old('trip_type') == 'زوجی' ? 'selected' : '' }}>زوجی</option>
                                <option value="خانوادگی" {{ old('trip_type') == 'خانوادگی' ? 'selected' : '' }}>خانوادگی</option>
                                <option value="دوستان" {{ old('trip_type') == 'دوستان' ? 'selected' : '' }}>با دوستان</option>
                                <option value="کاری" {{ old('trip_type') == 'کاری' ? 'selected' : '' }}>کاری</option>
                            </select>
                            @error('trip_type')
                            <div class="text-danger mt-2">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Travel Method -->
                        <div class="modern-form-group">
                            <label class="modern-label">وسیله سفر</label>
                            <select class="modern-input modern-select" id="travel_method" name="travel_method">
                                <option value="">انتخاب کنید</option>
                                <option value="اتومبیل شخصی" {{ old('travel_method') == 'اتومبیل شخصی' ? 'selected' : '' }}>اتومبیل شخصی</option>
                                <option value="هواپیما" {{ old('travel_method') == 'هواپیما' ? 'selected' : '' }}>هواپیما</option>
                                <option value="قطار" {{ old('travel_method') == 'قطار' ? 'selected' : '' }}>قطار</option>
                                <option value="اتوبوس" {{ old('travel_method') == 'اتوبوس' ? 'selected' : '' }}>اتوبوس</option>
                                <option value="تور" {{ old('travel_method') == 'تور' ? 'selected' : '' }}>تور</option>
                            </select>
                            @error('travel_method')
                            <div class="text-danger mt-2">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Stay Date -->
                        <div class="modern-form-group">
                            <label class="modern-label">{{ app()->getLocale() === 'fa' ? 'تاریخ اقامت' : 'Stay Date' }}</label>
                            <input type="date"
                                   class="modern-input"
                                   id="stay_date"
                                   name="stay_date"
                                   value="{{ old('stay_date') }}"
                                   placeholder="{{ app()->getLocale() === 'fa' ? 'YYYY/MM/DD' : 'YYYY-MM-DD' }}"
                                   data-locale-setup="true">
                            @error('stay_date')
                            <div class="text-danger mt-2">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Form Info -->
                <div class="form-info">
                    <i class="fas fa-info-circle form-info-icon"></i>
                    <div class="form-info-text">
                        نظر شما پس از بررسی و تایید مدیر در سایت نمایش داده خواهد شد. لطفاً نظرات سازنده و مفید ارائه دهید.
                    </div>
                </div>

                <!-- Form Actions -->
                <div class="form-actions">
                    <button type="button" onclick="window.history.back()" class="modern-btn modern-btn-secondary">
                        <i class="fas fa-times"></i>
                        انصراف
                    </button>

                    <button type="button" onclick="ratingCreateManager.clearForm()" class="modern-btn modern-btn-warning">
                        <i class="fas fa-eraser"></i>
                        پاک کردن فرم
                    </button>

                    <button type="submit" class="modern-btn" id="submitBtn">
                        <i class="fas fa-paper-plane"></i>
                        ثبت نظر
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script src="{{ asset('js/persian-date-picker.js') }}"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Modern Rating Create Manager
    class ModernRatingCreateManager {
        constructor() {
            this.init();
        }

        init() {
            this.initializeLocationSearch();
            this.initializeRatingStars();
            this.initializeCharacterCounters();
            this.initializeFormValidation();
            this.initializeAutoSave();
            this.updateRatingDisplay();
        }

        // Initialize Location Search
        initializeLocationSearch() {
            const searchInput = document.getElementById('locationSearch');
            const dropdown = document.getElementById('locationDropdown');
            const selectedLocationId = document.getElementById('selectedLocationId');

            if (!searchInput) return;

            let searchTimeout;

            searchInput.addEventListener('input', (e) => {
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(() => {
                    this.filterLocations(e.target.value);
                }, 300);
            });

            searchInput.addEventListener('focus', () => {
                dropdown.classList.add('show');
            });

            document.addEventListener('click', (e) => {
                if (!e.target.closest('.location-selector')) {
                    dropdown.classList.remove('show');
                }
            });

            // Location option click handlers
            dropdown.addEventListener('click', (e) => {
                const option = e.target.closest('.location-option');
                if (option) {
                    const id = option.dataset.id;
                    const title = option.dataset.title;
                    const category = option.dataset.category;

                    this.selectLocationFromSearch(id, title, category);
                }
            });
        }

        // Filter Locations
        filterLocations(query) {
            const dropdown = document.getElementById('locationDropdown');
            const options = dropdown.querySelectorAll('.location-option');

            options.forEach(option => {
                const title = option.dataset.title.toLowerCase();
                const category = option.dataset.category.toLowerCase();
                const searchQuery = query.toLowerCase();

                if (title.includes(searchQuery) || category.includes(searchQuery)) {
                    option.style.display = 'block';
                } else {
                    option.style.display = 'none';
                }
            });

            dropdown.classList.add('show');
        }

        // Select Location from Search
        selectLocationFromSearch(id, title, category) {
            const searchInput = document.getElementById('locationSearch');
            const dropdown = document.getElementById('locationDropdown');
            const selectedLocationId = document.getElementById('selectedLocationId');

            searchInput.value = `${title} (${category})`;
            selectedLocationId.value = id;
            dropdown.classList.remove('show');

            this.showNotification('مکان انتخاب شد', 'success');
        }

        // Initialize Rating Stars
        initializeRatingStars() {
            // Overall rating
            this.initializeStarGroup('rating', 'ratingValue', 'ratingText');

            // Detailed ratings
            this.initializeDetailedRatings();
        }

        // Initialize Star Group
        initializeStarGroup(name, valueElementId, textElementId) {
            const inputs = document.querySelectorAll(`input[name="${name}"]`);
            const labels = document.querySelectorAll(`label[for^="${name.replace('_', '-')}-"]`);
            const valueElement = document.getElementById(valueElementId);
            const textElement = document.getElementById(textElementId);

            // Add click and hover events to labels
            labels.forEach((label, index) => {
                const starValue = index + 1;

                // Click event
                label.addEventListener('click', () => {
                    const input = document.getElementById(label.getAttribute('for'));
                    if (input) {
                        input.checked = true;
                        this.updateStarDisplay(labels, starValue);

                        if (valueElement) {
                            valueElement.textContent = `${starValue}/5`;
                        }
                        if (textElement) {
                            textElement.textContent = this.getRatingText(starValue);
                        }
                    }
                });

                // Hover events
                label.addEventListener('mouseenter', () => {
                    this.updateStarDisplay(labels, starValue);
                });

                label.addEventListener('mouseleave', () => {
                    const checkedInput = document.querySelector(`input[name="${name}"]:checked`);
                    const checkedValue = checkedInput ? parseInt(checkedInput.value) : 0;
                    this.updateStarDisplay(labels, checkedValue);
                });
            });

            // Handle input change events for form validation
            inputs.forEach(input => {
                input.addEventListener('change', () => {
                    const value = parseInt(input.value);
                    this.updateStarDisplay(labels, value);

                    if (valueElement) {
                        valueElement.textContent = `${value}/5`;
                    }
                    if (textElement) {
                        textElement.textContent = this.getRatingText(value);
                    }
                });
            });
        }

        // Update Star Display
        updateStarDisplay(labels, rating) {
            labels.forEach((label, index) => {
                const star = label.querySelector('i');
                if (star) {
                    if (index < rating) {
                        star.style.color = '#fbbf24';
                        star.style.transform = 'scale(1.1)';
                    } else {
                        star.style.color = '#e5e7eb';
                        star.style.transform = 'scale(1)';
                    }
                }
            });
        }

        // Initialize Detailed Ratings
        initializeDetailedRatings() {
            const ratingTypes = [
                { name: 'service_rating', prefix: 'service' },
                { name: 'cleanliness_rating', prefix: 'cleanliness' },
                { name: 'location_rating', prefix: 'location' },
                { name: 'price_rating', prefix: 'price' }
            ];

            ratingTypes.forEach(ratingType => {
                const inputs = document.querySelectorAll(`input[name="${ratingType.name}"]`);
                const labels = document.querySelectorAll(`label[for^="${ratingType.prefix}-"]`);

                // Add click and hover events to labels
                labels.forEach((label, index) => {
                    const starValue = index + 1;

                    // Click event
                    label.addEventListener('click', () => {
                        const input = document.getElementById(label.getAttribute('for'));
                        if (input) {
                            input.checked = true;
                            this.updateStarDisplay(labels, starValue);
                        }
                    });

                    // Hover events
                    label.addEventListener('mouseenter', () => {
                        this.updateStarDisplay(labels, starValue);
                    });

                    label.addEventListener('mouseleave', () => {
                        const checkedInput = document.querySelector(`input[name="${ratingType.name}"]:checked`);
                        const checkedValue = checkedInput ? parseInt(checkedInput.value) : 0;
                        this.updateStarDisplay(labels, checkedValue);
                    });
                });

                // Handle input change events
                inputs.forEach(input => {
                    input.addEventListener('change', () => {
                        const value = parseInt(input.value);
                        this.updateStarDisplay(labels, value);
                        this.updateDetailedRatingDisplay(ratingType.name, value);
                    });
                });
            });
        }

        // Update Detailed Rating Display
        updateDetailedRatingDisplay(type, value) {
            // You can add visual feedback here if needed
            console.log(`${type} rated: ${value}`);
        }

        // Get Rating Text
        getRatingText(rating) {
            const texts = {
                '1': 'خیلی ضعیف',
                '2': 'ضعیف',
                '3': 'متوسط',
                '4': 'خوب',
                '5': 'عالی'
            };
            return texts[rating] || 'امتیاز خود را انتخاب کنید';
        }

        // Initialize Character Counters
        initializeCharacterCounters() {
            const positiveComment = document.getElementById('positive_comment');
            const negativeComment = document.getElementById('negative_comment');
            const positiveCount = document.getElementById('positiveCount');
            const negativeCount = document.getElementById('negativeCount');

            if (positiveComment && positiveCount) {
                positiveComment.addEventListener('input', () => {
                    positiveCount.textContent = positiveComment.value.length;
                });
                // Initialize count
                positiveCount.textContent = positiveComment.value.length;
            }

            if (negativeComment && negativeCount) {
                negativeComment.addEventListener('input', () => {
                    negativeCount.textContent = negativeComment.value.length;
                });
                // Initialize count
                negativeCount.textContent = negativeComment.value.length;
            }
        }

        // Initialize Form Validation
        initializeFormValidation() {
            const form = document.getElementById('ratingForm');
            const submitBtn = document.getElementById('submitBtn');

            form.addEventListener('submit', (e) => {
                if (!this.validateForm()) {
                    e.preventDefault();
                    return false;
                }

                // Clear draft before submission
                localStorage.removeItem('rating_draft');

                // Show loading state
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> در حال ثبت...';
                submitBtn.disabled = true;
            });
        }

        // Validate Form
        validateForm() {
            const locationId = document.getElementById('selectedLocationId').value;
            const rating = document.querySelector('input[name="rating"]:checked');

            if (!locationId) {
                this.showNotification('لطفاً مکان را انتخاب کنید', 'error');
                return false;
            }

            if (!rating) {
                this.showNotification('لطفاً امتیاز کلی را انتخاب کنید', 'error');
                return false;
            }

            return true;
        }

        // Initialize Auto Save
        initializeAutoSave() {
            const form = document.getElementById('ratingForm');
            const inputs = form.querySelectorAll('input, select, textarea');

            inputs.forEach(input => {
                input.addEventListener('change', () => {
                    this.autoSave();
                });
            });
        }

        // Auto Save
        autoSave() {
            const formData = new FormData(document.getElementById('ratingForm'));
            const data = {};

            for (let [key, value] of formData.entries()) {
                data[key] = value;
            }

            localStorage.setItem('rating_draft', JSON.stringify(data));
        }

        // Load Draft
        loadDraft() {
            const draft = localStorage.getItem('rating_draft');
            if (draft) {
                const data = JSON.parse(draft);

                Object.keys(data).forEach(key => {
                    const element = document.querySelector(`[name="${key}"]`);
                    if (element) {
                        if (element.type === 'radio') {
                            const radio = document.querySelector(`[name="${key}"][value="${data[key]}"]`);
                            if (radio) radio.checked = true;
                        } else {
                            element.value = data[key];
                        }
                    }
                });

                this.updateRatingDisplay();
                this.initializeCharacterCounters();
            }
        }

        // Update Rating Display
        updateRatingDisplay() {
            // Update main rating
            const ratingInput = document.querySelector('input[name="rating"]:checked');
            if (ratingInput) {
                const value = parseInt(ratingInput.value);
                const valueElement = document.getElementById('ratingValue');
                const textElement = document.getElementById('ratingText');
                const labels = document.querySelectorAll('label[for^="rating-"]');

                if (valueElement) valueElement.textContent = `${value}/5`;
                if (textElement) textElement.textContent = this.getRatingText(value);
                this.updateStarDisplay(labels, value);
            }

            // Update detailed ratings
            const ratingTypes = [
                { name: 'service_rating', prefix: 'service' },
                { name: 'cleanliness_rating', prefix: 'cleanliness' },
                { name: 'location_rating', prefix: 'location' },
                { name: 'price_rating', prefix: 'price' }
            ];

            ratingTypes.forEach(ratingType => {
                const checkedInput = document.querySelector(`input[name="${ratingType.name}"]:checked`);
                if (checkedInput) {
                    const value = parseInt(checkedInput.value);
                    const labels = document.querySelectorAll(`label[for^="${ratingType.prefix}-"]`);
                    this.updateStarDisplay(labels, value);
                }
            });
        }

        // Show Notification
        showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `notification notification-${type}`;
            notification.innerHTML = `
                <div class="notification-content">
                    <i class="fas fa-${this.getNotificationIcon(type)}"></i>
                    <span>${message}</span>
                </div>
                <button class="notification-close" onclick="this.parentElement.remove()">
                    <i class="fas fa-times"></i>
                </button>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                if (notification.parentElement) {
                    notification.remove();
                }
            }, 5000);
        }

        getNotificationIcon(type) {
            switch (type) {
                case 'success': return 'check-circle';
                case 'error': return 'exclamation-circle';
                case 'warning': return 'exclamation-triangle';
                default: return 'info-circle';
            }
        }

        // Clear Form
        clearForm() {
            const form = document.getElementById('ratingForm');

            // Reset all form fields
            form.reset();

            // Clear location selection
            const locationSearch = document.getElementById('locationSearch');
            const selectedLocationId = document.getElementById('selectedLocationId');
            if (locationSearch) locationSearch.value = '';
            if (selectedLocationId) selectedLocationId.value = '';

            // Reset star displays
            const allStarLabels = document.querySelectorAll('.star-label, .detailed-star-label');
            allStarLabels.forEach(label => {
                const star = label.querySelector('i');
                if (star) {
                    star.style.color = '#e5e7eb';
                    star.style.transform = 'scale(1)';
                }
            });

            // Reset rating displays
            const ratingValue = document.getElementById('ratingValue');
            const ratingText = document.getElementById('ratingText');
            if (ratingValue) ratingValue.textContent = '0/5';
            if (ratingText) ratingText.textContent = 'امتیاز انتخاب نشده';

            // Reset character counters
            const positiveCount = document.getElementById('positiveCount');
            const negativeCount = document.getElementById('negativeCount');
            if (positiveCount) positiveCount.textContent = '0';
            if (negativeCount) negativeCount.textContent = '0';

            // Clear localStorage
            localStorage.removeItem('rating_draft');

            // Show notification
            this.showNotification('فرم پاک شد', 'success');
        }
    }

    // Initialize Rating Create Manager
    window.ratingCreateManager = new ModernRatingCreateManager();

    // Load draft on page load
    ratingCreateManager.loadDraft();

    // Check for success message and clear form if exists
    @if(session('success'))
        // Clear form after successful submission
        setTimeout(() => {
            ratingCreateManager.clearForm();
        }, 1000);
    @endif

    // Global Functions
    window.selectLocation = function(id, title, category) {
        const searchInput = document.getElementById('locationSearch');
        const selectedLocationId = document.getElementById('selectedLocationId');

        if (searchInput) {
            searchInput.value = `${title} (${category})`;
        }
        if (selectedLocationId) {
            selectedLocationId.value = id;
        }

        ratingCreateManager.showNotification('مکان انتخاب شد', 'success');
    };

    window.changeLocation = function() {
        const selectedLocation = document.querySelector('.selected-location');
        const locationSelector = document.querySelector('.location-selector');

        if (selectedLocation) {
            selectedLocation.style.display = 'none';
        }
        if (locationSelector) {
            locationSelector.style.display = 'block';
        }
    };

    // Clear draft on successful submission
    window.addEventListener('beforeunload', () => {
        // Only clear if form was submitted successfully
        if (document.querySelector('.alert-success')) {
            localStorage.removeItem('rating_draft');
        }
    });
});
</script>

<!-- Notification Styles -->
<style>
.notification {
    position: fixed;
    top: 20px;
    right: 20px;
    background: white;
    border-radius: 12px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
    padding: 1rem 1.5rem;
    display: flex;
    align-items: center;
    justify-content: space-between;
    z-index: 10000;
    min-width: 300px;
    animation: slideInRight 0.3s ease;
}

.notification-success {
    border-left: 4px solid #10b981;
}

.notification-error {
    border-left: 4px solid #ef4444;
}

.notification-warning {
    border-left: 4px solid #f59e0b;
}

.notification-info {
    border-left: 4px solid #3b82f6;
}

.notification-content {
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.notification-content i {
    font-size: 1.25rem;
}

.notification-success .notification-content i {
    color: #10b981;
}

.notification-error .notification-content i {
    color: #ef4444;
}

.notification-warning .notification-content i {
    color: #f59e0b;
}

.notification-info .notification-content i {
    color: #3b82f6;
}

.notification-close {
    background: none;
    border: none;
    color: #6b7280;
    cursor: pointer;
    padding: 0.25rem;
    border-radius: 4px;
    transition: all 0.2s ease;
}

.notification-close:hover {
    background: #f3f4f6;
    color: #374151;
}

@keyframes slideInRight {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}
</style>
@endsection