@extends('admin.layouts.app')

@section('title', 'افزودن استان جدید')

@section('content')
<div class="container-fluid">
    <div class="card">
        <div class="card-header">
            <div class="d-flex justify-content-between align-items-center">
                <h4>افزودن استان جدید</h4>
                <a href="{{ route('admin.geography.provinces.index') }}" class="btn btn-primary">
                    <i class="fas fa-list"></i> بازگشت به لیست استان‌ها
                </a>
            </div>
        </div>
        <div class="card-body">
            @if(session('error'))
                <div class="alert alert-danger">
                    {{ session('error') }}
                </div>
            @endif

            @if ($errors->any())
                <div class="alert alert-danger">
                    <ul class="mb-0">
                        @foreach ($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            @endif

            <form action="{{ route('admin.geography.provinces.store') }}" method="POST">
                @csrf
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group mb-3">
                            <label for="country_id" class="form-label">کشور <span class="text-danger">*</span></label>
                            <select name="country_id" id="country_id" class="form-control country-flag-select @error('country_id') is-invalid @enderror" required>
                                <option value="">انتخاب کشور</option>
                                @foreach($countries as $id => $name)
                                    @php
                                        $flag = '';
                                        if (is_object($name) && isset($name->flag)) $flag = $name->flag;
                                        elseif (is_array($name) && isset($name['flag'])) $flag = $name['flag'];
                                    @endphp
                                    <option value="{{ $id }}" {{ old('country_id') == $id ? 'selected' : '' }}>
                                        {{ $flag }} {{ is_object($name) ? $name->name : (is_array($name) ? $name['name'] : $name) }}
                                    </option>
                                @endforeach
                            </select>
                            @error('country_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group mb-3">
                            <label for="name" class="form-label">نام استان <span class="text-danger">*</span></label>
                            <input type="text" class="form-control @error('name') is-invalid @enderror" id="name" name="name" value="{{ old('name') }}" required>
                            @error('name')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group mb-3">
                            <label for="name_en" class="form-label">نام انگلیسی</label>
                            <input type="text" class="form-control @error('name_en') is-invalid @enderror" id="name_en" name="name_en" value="{{ old('name_en') }}">
                            @error('name_en')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group mb-3">
                            <label for="slug" class="form-label">نامک (slug) <span class="text-danger">*</span></label>
                            <input type="text" class="form-control @error('slug') is-invalid @enderror" id="slug" name="slug" value="{{ old('slug') }}" required>
                            @error('slug')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <small class="form-text text-muted">نامک در URL استفاده می‌شود و فقط شامل حروف، اعداد و خط تیره باشد.</small>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group mb-3">
                            <label for="center_city_name" class="form-label">نام شهر مرکزی</label>
                            <input type="text" class="form-control @error('center_city_name') is-invalid @enderror" id="center_city_name" name="center_city_name" value="{{ old('center_city_name') }}">
                            @error('center_city_name')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group mb-3">
                            <label for="code" class="form-label">کد استان</label>
                            <input type="text" class="form-control @error('code') is-invalid @enderror" id="code" name="code" value="{{ old('code') }}">
                            @error('code')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group mb-3">
                            <label class="form-label d-block">وضعیت</label>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="radio" name="is_active" id="status_active" value="1" {{ old('is_active', '1') == '1' ? 'checked' : '' }}>
                                <label class="form-check-label" for="status_active">فعال</label>
                            </div>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="radio" name="is_active" id="status_inactive" value="0" {{ old('is_active') == '0' ? 'checked' : '' }}>
                                <label class="form-check-label" for="status_inactive">غیرفعال</label>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="row mt-3">
                    <div class="col-md-12">
                        <div class="form-group mb-3">
                            <label for="boundaries" class="form-label">مرزهای جغرافیایی (GeoJSON)</label>
                            <div class="mb-2">
                                <button type="button" id="insert-template" class="btn btn-sm btn-outline-secondary">Insert GeoJSON Template</button>
                                <a href="https://geojson.io/" target="_blank" class="btn btn-sm btn-outline-info">Create GeoJSON Online</a>
                                <button type="button" id="fix-json" class="btn btn-sm btn-outline-danger">Fix Corrupted Data</button>
                            </div>
                            <div class="alert alert-info mb-2">
                                <i class="fas fa-info-circle"></i> مرزهای جغرافیایی باید در فرمت GeoJSON و با نوع <code>Polygon</code> وارد شوند.
                            </div>
                            <textarea class="form-control @error('boundaries') is-invalid @enderror" id="boundaries" name="boundaries" rows="10">{{ old('boundaries') }}</textarea>
                            <small class="form-text text-muted">دیتای GeoJSON برای نمایش مرزهای استان روی نقشه را وارد کنید</small>
                            @error('boundaries')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
                
                <div class="mt-4">
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-save"></i> ذخیره استان
                    </button>
                    <a href="{{ route('admin.geography.provinces.index') }}" class="btn btn-secondary">
                        <i class="fas fa-times"></i> انصراف
                    </a>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    $(document).ready(function() {
        // تولید خودکار نامک از روی نام استان
        $('#name').on('keyup', function() {
            const name = $(this).val();
            const slug = name.toLowerCase()
                .replace(/\s+/g, '-')           // Replace spaces with -
                .replace(/[^\w\-]+/g, '')       // Remove all non-word chars
                .replace(/\-\-+/g, '-')         // Replace multiple - with single -
                .replace(/^-+/, '')             // Trim - from start of text
                .replace(/-+$/, '');            // Trim - from end of text
            
            $('#slug').val(slug);
        });
        
        // Insert GeoJSON template
        $('#insert-template').on('click', function() {
            // Simple Polygon format - this format matches existing data in the database
            const template = {
                "type": "Polygon",
                "coordinates": [
                    [
                        [51.3, 35.7], // Tehran area coordinates
                        [51.5, 35.7],
                        [51.5, 35.8],
                        [51.3, 35.8],
                        [51.3, 35.7]
                    ]
                ]
            };
            
            $('#boundaries').val(JSON.stringify(template, null, 2));
        });
        
        // Fix corrupted JSON data
        $('#fix-json').on('click', function() {
            var currentValue = $('#boundaries').val();
            
            // If empty or null, just insert the template
            if (!currentValue || currentValue === 'null') {
                $('#insert-template').click();
                return;
            }
            
            // Check if it looks corrupted
            var isBadData = false;
            
            // Look for JSON parse errors
            try {
                JSON.parse(currentValue);
            } catch(e) {
                isBadData = true;
            }
            
            // Non-printable characters suggest binary data
            for (var i = 0; i < currentValue.length; i++) {
                var code = currentValue.charCodeAt(i);
                if (code < 32 || code === 127) {
                    isBadData = true;
                    break;
                }
            }
            
            if (isBadData) {
                if (confirm('داده‌های موجود معتبر نیستند. آیا می‌خواهید از الگوی پیش‌فرض استفاده کنید؟')) {
                    $('#insert-template').click();
                }
                return;
            }
            
            try {
                // Try to pretty-print it
                const parsed = JSON.parse(currentValue);
                $('#boundaries').val(JSON.stringify(parsed, null, 2));
                alert('داده‌های GeoJSON با موفقیت تمیز شدند.');
            } catch (e) {
                alert('خطا: ' + e.message);
            }
        });
    });
</script>
@endsection 