@php
    use App\Models\Location;
    use App\Models\Category;
    use App\Models\Rating;
    use App\Models\User;
    use App\Models\Admin;
    use App\Models\Advertisement;
    use Illuminate\Support\Facades\DB;
    use Illuminate\Support\Facades\Cache;

    // دریافت آمار پویا از دیتابیس با کش برای بهبود عملکرد
    $sidebarStats = Cache::remember('sidebar_stats', 300, function () {
        try {
            return [
                'total_locations' => Location::count(),
                'active_locations' => Location::where('is_active', 1)->count(),
                'pending_locations' => Location::where('is_active', 0)->count(),
                'total_categories' => Category::count(),
                'total_ratings' => Rating::count(),
                'pending_ratings' => Rating::where('status', 'pending')->count(),
                'approved_ratings' => Rating::where('status', 'approved')->count(),
                'total_users' => User::count(),
                'active_users' => User::where('status', 'active')->count(), // Fixed: use status column
                'inactive_users' => User::where('status', 'inactive')->count(),
                'banned_users' => User::where('status', 'banned')->count(),
                'total_admins' => Admin::count(),
                'active_admins' => Admin::where('is_active', 1)->count(),
                'total_ads' => Advertisement::count(),
                'active_ads' => Advertisement::where('is_active', 1)->count(),
                'unread_messages' => 0, // TODO: Implement message system
                'system_notifications' => 0, // TODO: Implement notification system
            ];
        } catch (\Exception $e) {
            // Fallback values in case of database errors
            return [
                'total_locations' => 0,
                'active_locations' => 0,
                'pending_locations' => 0,
                'total_categories' => 0,
                'total_ratings' => 0,
                'pending_ratings' => 0,
                'approved_ratings' => 0,
                'total_users' => 0,
                'active_users' => 0,
                'inactive_users' => 0,
                'banned_users' => 0,
                'total_admins' => 0,
                'active_admins' => 0,
                'total_ads' => 0,
                'active_ads' => 0,
                'unread_messages' => 0,
                'system_notifications' => 0,
            ];
        }
    });

    // دسته‌بندی آیتم‌های منو با اطلاعات پویا
    $menuStructure = [
        [
            'title' => 'اصلی',
            'items' => [
                [
                    'icon' => 'tachometer-alt',
                    'text' => 'داشبورد',
                    'route' => 'admin.dashboard',
                    'active' => request()->routeIs('admin.dashboard'),
                    'badge' => [
                        'text' => 'مرکز کنترل',
                        'type' => 'primary'
                    ]
                ],
            ]
        ],
        [
            'title' => 'مدیریت محتوا',
            'items' => [
                [
                    'icon' => 'map-marker-alt',
                    'text' => 'مدیریت مکان‌ها',
                    'route' => '#',
                    'active' => request()->routeIs('admin.locations.*'),
                    'badge' => [
                        'text' => $sidebarStats['total_locations'],
                        'type' => 'success'
                    ],
                    'submenu' => [
                        [
                            'icon' => 'list',
                            'text' => 'لیست مکان‌ها',
                            'route' => 'admin.locations.index',
                            'active' => request()->routeIs('admin.locations.index'),
                            'badge' => [
                                'text' => $sidebarStats['active_locations'],
                                'type' => 'info'
                            ]
                        ],
                        [
                            'icon' => 'plus',
                            'text' => 'افزودن مکان جدید',
                            'route' => 'admin.locations.create',
                            'active' => request()->routeIs('admin.locations.create')
                        ],
                        [
                            'icon' => 'clock',
                            'text' => 'مکان‌های غیرفعال',
                            'route' => 'admin.locations.index',
                            'active' => request()->routeIs('admin.locations.index') && request()->get('status') == 'inactive',
                            'badge' => [
                                'text' => $sidebarStats['pending_locations'],
                                'type' => $sidebarStats['pending_locations'] > 0 ? 'warning' : 'secondary'
                            ]
                        ],
                        [
                            'icon' => 'tags',
                            'text' => 'برچسب‌ها',
                            'route' => 'admin.tags.index',
                            'active' => request()->routeIs('admin.tags.*')
                        ],
                    ]
                ],
                [
                    'icon' => 'folder',
                    'text' => 'دسته‌بندی‌ها',
                    'route' => '#',
                    'active' => request()->routeIs('admin.categories.*'),
                    'badge' => [
                        'text' => $sidebarStats['total_categories'],
                        'type' => 'info'
                    ],
                    'submenu' => [
                        [
                            'icon' => 'list',
                            'text' => 'لیست دسته‌بندی‌ها',
                            'route' => 'admin.categories.index',
                            'active' => request()->routeIs('admin.categories.index'),
                            'badge' => [
                                'text' => $sidebarStats['total_categories'],
                                'type' => 'primary'
                            ]
                        ],
                        [
                            'icon' => 'plus',
                            'text' => 'افزودن دسته‌بندی',
                            'route' => 'admin.categories.create',
                            'active' => request()->routeIs('admin.categories.create')
                        ],
                    ]
                ],
                [
                    'icon' => 'check-circle',
                    'text' => 'امکانات',
                    'route' => '#',
                    'active' => request()->routeIs('admin.amenities.*') || request()->routeIs('admin.amenity-types.*'),
                    'submenu' => [
                        [
                            'icon' => 'list',
                            'text' => 'انواع امکانات',
                            'route' => 'admin.amenity-types.index',
                            'active' => request()->routeIs('admin.amenity-types.*')
                        ],
                        [
                            'icon' => 'check',
                            'text' => 'امکانات',
                            'route' => 'admin.amenities.index',
                            'active' => request()->routeIs('admin.amenities.index')
                        ],
                    ]
                ],
                [
                    'icon' => 'star',
                    'text' => 'نظرات و امتیازات',
                    'route' => '#',
                    'active' => request()->routeIs('admin.ratings.*'),
                    'badge' => [
                        'text' => $sidebarStats['total_ratings'],
                        'type' => 'warning'
                    ],
                    'submenu' => [
                        [
                            'icon' => 'list',
                            'text' => 'لیست نظرات',
                            'route' => 'admin.ratings.index',
                            'active' => request()->routeIs('admin.ratings.index'),
                            'badge' => [
                                'text' => $sidebarStats['approved_ratings'],
                                'type' => 'success'
                            ]
                        ],
                        [
                            'icon' => 'check-double',
                            'text' => 'نظرات تایید نشده',
                            'route' => 'admin.ratings.pending',
                            'active' => request()->routeIs('admin.ratings.pending'),
                            'badge' => [
                                'text' => $sidebarStats['pending_ratings'],
                                'type' => $sidebarStats['pending_ratings'] > 0 ? 'danger' : 'secondary'
                            ]
                        ],
                    ]
                ],
                [
                    'icon' => 'ad',
                    'text' => 'مدیریت تبلیغات',
                    'route' => '#',
                    'active' => request()->routeIs('admin.advertisements.*') || request()->routeIs('admin.ads.*'),
                    'badge' => [
                        'text' => $sidebarStats['total_ads'],
                        'type' => 'info'
                    ],
                    'submenu' => [
                        [
                            'icon' => 'list',
                            'text' => 'لیست تبلیغات',
                            'route' => 'admin.advertisements.index',
                            'active' => request()->routeIs('admin.advertisements.index'),
                            'badge' => [
                                'text' => $sidebarStats['active_ads'],
                                'type' => 'success'
                            ]
                        ],
                        [
                            'icon' => 'plus',
                            'text' => 'افزودن تبلیغ',
                            'route' => 'admin.advertisements.create',
                            'active' => request()->routeIs('admin.advertisements.create')
                        ],
                        [
                            'icon' => 'th-large',
                            'text' => 'زون‌های نمایش',
                            'route' => 'admin.ads.display-zones',
                            'active' => request()->routeIs('admin.ads.display-zones')
                        ],
                        [
                            'icon' => 'object-group',
                            'text' => 'تخصیص آگهی به زون‌ها',
                            'route' => 'admin.ads.manage-zone-ads',
                            'active' => request()->routeIs('admin.ads.manage-zone-ads')
                        ],
                    ]
                ],
                [
                    'icon' => 'envelope',
                    'text' => 'پیام‌ها',
                    'route' => '#',
                    'active' => request()->routeIs('admin.messages.*'),
                    'submenu' => [
                        [
                            'icon' => 'inbox',
                            'text' => 'صندوق ورودی',
                            'route' => 'admin.messages.inbox',
                            'active' => request()->routeIs('admin.messages.inbox')
                        ],
                        [
                            'icon' => 'paper-plane',
                            'text' => 'پیام‌های ارسالی',
                            'route' => 'admin.messages.sent',
                            'active' => request()->routeIs('admin.messages.sent')
                        ],
                        [
                            'icon' => 'trash',
                            'text' => 'سطل زباله',
                            'route' => 'admin.messages.trash',
                            'active' => request()->routeIs('admin.messages.trash')
                        ],
                    ]
                ],
            ]
        ],
        [
            'title' => 'مدیریت کاربران',
            'items' => [
                [
                    'icon' => 'users',
                    'text' => 'مدیریت کاربران',
                    'route' => '#',
                    'active' => request()->routeIs('admin.users.*'),
                    'badge' => [
                        'text' => $sidebarStats['total_users'],
                        'type' => 'primary'
                    ],
                    'submenu' => [
                        [
                            'icon' => 'list',
                            'text' => 'لیست کاربران',
                            'route' => 'admin.users.index',
                            'active' => request()->routeIs('admin.users.index'),
                            'badge' => [
                                'text' => $sidebarStats['active_users'],
                                'type' => 'success'
                            ]
                        ],
                        [
                            'icon' => 'user-plus',
                            'text' => 'افزودن کاربر',
                            'route' => 'admin.users.create',
                            'active' => request()->routeIs('admin.users.create')
                        ],
                        [
                            'icon' => 'user-times',
                            'text' => 'کاربران غیرفعال',
                            'route' => 'admin.users.index',
                            'active' => request()->routeIs('admin.users.index') && request()->get('status') == 'inactive',
                            'badge' => [
                                'text' => $sidebarStats['inactive_users'],
                                'type' => $sidebarStats['inactive_users'] > 0 ? 'warning' : 'secondary'
                            ]
                        ],
                        [
                            'icon' => 'user-slash',
                            'text' => 'کاربران مسدود',
                            'route' => 'admin.users.index',
                            'active' => request()->routeIs('admin.users.index') && request()->get('status') == 'banned',
                            'badge' => [
                                'text' => $sidebarStats['banned_users'],
                                'type' => $sidebarStats['banned_users'] > 0 ? 'danger' : 'secondary'
                            ]
                        ],
                        [
                            'icon' => 'user-shield',
                            'text' => 'مدیریت مدیران',
                            'route' => 'admin.admins.index',
                            'active' => request()->routeIs('admin.admins.*'),
                            'badge' => [
                                'text' => $sidebarStats['active_admins'],
                                'type' => 'warning'
                            ]
                        ],
                    ]
                ],
                [
                    'icon' => 'user-lock',
                    'text' => 'سطوح دسترسی',
                    'route' => '#',
                    'active' => request()->routeIs('admin.roles.*') || request()->routeIs('admin.permissions.*'),
                    'submenu' => [
                        [
                            'icon' => 'user-tag',
                            'text' => 'نقش‌ها',
                            'route' => 'admin.roles.index',
                            'active' => request()->routeIs('admin.roles.*')
                        ],
                        [
                            'icon' => 'key',
                            'text' => 'دسترسی‌ها',
                            'route' => 'admin.permissions.index',
                            'active' => request()->routeIs('admin.permissions.*')
                        ],
                    ]
                ],
            ]
        ],
        [
            'title' => 'آمار و گزارشات',
            'items' => [
                [
                    'icon' => 'chart-bar',
                    'text' => 'آمار کلی',
                    'route' => 'admin.statistics.index',
                    'active' => request()->routeIs('admin.statistics.index'),
                    'badge' => [
                        'text' => 'جدید',
                        'type' => 'success'
                    ]
                ],
                [
                    'icon' => 'chart-line',
                    'text' => 'آمار بازدیدها',
                    'route' => 'admin.statistics.visits',
                    'active' => request()->routeIs('admin.statistics.visits')
                ],
                [
                    'icon' => 'analytics',
                    'text' => 'تحلیل‌های پیشرفته',
                    'route' => '#',
                    'active' => request()->routeIs('admin.analytics.*'),
                    'badge' => [
                        'text' => 'Pro',
                        'type' => 'primary'
                    ],
                    'submenu' => [
                        [
                            'icon' => 'chart-area',
                            'text' => 'تحلیل عملکرد',
                            'route' => 'admin.analytics.performance',
                            'active' => request()->routeIs('admin.analytics.performance')
                        ],
                        [
                            'icon' => 'chart-pie',
                            'text' => 'تحلیل پیش‌بینی',
                            'route' => 'admin.analytics.predictive',
                            'active' => request()->routeIs('admin.analytics.predictive')
                        ],
                        [
                            'icon' => 'globe',
                            'text' => 'تحلیل جغرافیایی',
                            'route' => 'admin.analytics.geographic',
                            'active' => request()->routeIs('admin.analytics.geographic')
                        ],
                        [
                            'icon' => 'users',
                            'text' => 'تحلیل کاربران',
                            'route' => 'admin.analytics.user',
                            'active' => request()->routeIs('admin.analytics.user')
                        ],
                        [
                            'icon' => 'chart-bar',
                            'text' => 'مقایسه‌ها',
                            'route' => 'admin.analytics.comparisons',
                            'active' => request()->routeIs('admin.analytics.comparisons')
                        ],
                    ]
                ],
                [
                    'icon' => 'chart-pie',
                    'text' => 'گزارش‌های پیشرفته',
                    'route' => '#',
                    'active' => request()->routeIs('admin.reports.*'),
                    'submenu' => [
                        [
                            'icon' => 'file-alt',
                            'text' => 'گزارش مکان‌ها',
                            'route' => 'admin.reports.locations',
                            'active' => request()->routeIs('admin.reports.locations')
                        ],
                        [
                            'icon' => 'file-alt',
                            'text' => 'گزارش کاربران',
                            'route' => 'admin.reports.users',
                            'active' => request()->routeIs('admin.reports.users')
                        ],
                        [
                            'icon' => 'file-alt',
                            'text' => 'گزارش تبلیغات',
                            'route' => 'admin.reports.advertisements',
                            'active' => request()->routeIs('admin.reports.advertisements')
                        ],
                    ]
                ],
            ]
        ],
        [
            'title' => 'تنظیمات سیستم',
            'items' => [
                [
                    'icon' => 'cog',
                    'text' => 'تنظیمات عمومی',
                    'route' => '#',
                    'active' => request()->routeIs('admin.settings.*'),
                    'submenu' => [
                        [
                            'icon' => 'cog',
                            'text' => 'تنظیمات پایه',
                            'route' => 'admin.settings.general',
                            'active' => request()->routeIs('admin.settings.general')
                        ],
                        [
                            'icon' => 'envelope',
                            'text' => 'تنظیمات ایمیل',
                            'route' => 'admin.settings.email',
                            'active' => request()->routeIs('admin.settings.email')
                        ],
                        [
                            'icon' => 'credit-card',
                            'text' => 'تنظیمات پرداخت',
                            'route' => 'admin.settings.payment',
                            'active' => request()->routeIs('admin.settings.payment')
                        ],
                    ]
                ],
                [
                    'icon' => 'desktop',
                    'text' => 'تنظیمات نمایش',
                    'route' => '#',
                    'active' => request()->routeIs('admin.display-settings.*') || request()->routeIs('admin.templates.*'),
                    'submenu' => [
                        [
                            'icon' => 'desktop',
                            'text' => 'تنظیمات نمایش',
                            'route' => 'admin.display-settings.index',
                            'active' => request()->routeIs('admin.display-settings.index')
                        ],
                        [
                            'icon' => 'layer-group',
                            'text' => 'مدیریت قالب‌ها',
                            'route' => 'admin.category-templates.index',
                            'active' => request()->routeIs('admin.category-templates.*')
                        ],
                        [
                            'icon' => 'sliders-h',
                            'text' => 'تنظیمات صفحه اصلی',
                            'route' => 'admin.display-settings.homepage',
                            'active' => request()->routeIs('admin.display-settings.homepage')
                        ],
                    ]
                ],
                [
                    'icon' => 'bars',
                    'text' => 'مدیریت منو و هدر',
                    'route' => '#',
                    'active' => request()->routeIs('admin.menu-management.*') || request()->routeIs('admin.header-settings.*'),
                    'submenu' => [
                        [
                            'icon' => 'bars',
                            'text' => 'مدیریت منوها',
                            'route' => 'admin.menu-management.index',
                            'active' => request()->routeIs('admin.menu-management.*')
                        ],
                        [
                            'icon' => 'window-maximize',
                            'text' => 'تنظیمات هدر',
                            'route' => 'admin.header-settings.index',
                            'active' => request()->routeIs('admin.header-settings.*')
                        ],
                    ]
                ],
                [
                    'icon' => 'shield-alt',
                    'text' => 'تنظیمات امنیتی',
                    'route' => '#',
                    'active' => request()->routeIs('admin.security-settings.*'),
                    'submenu' => [
                        [
                            'icon' => 'shield-alt',
                            'text' => 'تنظیمات امنیتی',
                            'route' => 'admin.security-settings.index',
                            'active' => request()->routeIs('admin.security-settings.index')
                        ],
                        [
                            'icon' => 'history',
                            'text' => 'تاریخچه ورود',
                            'route' => 'admin.security-settings.login-history',
                            'active' => request()->routeIs('admin.security-settings.login-history')
                        ],
                    ]
                ],
                [
                    'icon' => 'language',
                    'text' => 'زبان‌ها و ترجمه‌ها',
                    'route' => '#',
                    'active' => request()->routeIs('admin.languages.*') || request()->routeIs('admin.translations.*'),
                    'submenu' => [
                        [
                            'icon' => 'language',
                            'text' => 'زبان‌ها',
                            'route' => 'admin.languages.index',
                            'active' => request()->routeIs('admin.languages.*')
                        ],
                        [
                            'icon' => 'translate',
                            'text' => 'ترجمه‌ها',
                            'route' => 'admin.translations.index',
                            'active' => request()->routeIs('admin.translations.*')
                        ],
                    ]
                ],
                [
                    'icon' => 'cogs',
                    'text' => 'تنظیمات ماژول‌ها',
                    'route' => '#',
                    'active' => request()->routeIs('admin.modules.*'),
                    'submenu' => [
                        [
                            'icon' => 'plug',
                            'text' => 'ماژول‌های فعال',
                            'route' => 'admin.modules.active',
                            'active' => request()->routeIs('admin.modules.active')
                        ],
                        [
                            'icon' => 'plus-circle',
                            'text' => 'نصب ماژول جدید',
                            'route' => 'admin.modules.install',
                            'active' => request()->routeIs('admin.modules.install')
                        ],
                    ]
                ],
                [
                    'icon' => 'globe',
                    'text' => 'اطلاعات جغرافیایی',
                    'route' => '#',
                    'active' => request()->routeIs('admin.geography.*'),
                    'submenu' => [
                        [
                            'icon' => 'flag',
                            'text' => 'کشورها',
                            'route' => 'admin.geography.countries.index',
                            'active' => request()->routeIs('admin.geography.countries.*')
                        ],
                        [
                            'icon' => 'map',
                            'text' => 'استان‌ها',
                            'route' => 'admin.geography.provinces.index',
                            'active' => request()->routeIs('admin.geography.provinces.*')
                        ],
                        [
                            'icon' => 'map-marked',
                            'text' => 'شهرستان‌ها',
                            'route' => 'admin.geography.counties.index',
                            'active' => request()->routeIs('admin.geography.counties.*')
                        ],
                        [
                            'icon' => 'draw-polygon',
                            'text' => 'بخش‌ها',
                            'route' => 'admin.geography.districts.index',
                            'active' => request()->routeIs('admin.geography.districts.*')
                        ],
                        [
                            'icon' => 'city',
                            'text' => 'شهرها',
                            'route' => 'admin.geography.cities.index',
                            'active' => request()->routeIs('admin.geography.cities.*')
                        ],
                        [
                            'icon' => 'village',
                            'text' => 'روستاها',
                            'route' => 'admin.geography.villages.index',
                            'active' => request()->routeIs('admin.geography.villages.*')
                        ],
                        [
                            'icon' => 'vector-square',
                            'text' => 'مناطق',
                            'route' => 'admin.geography.zones.index',
                            'active' => request()->routeIs('admin.geography.zones.*')
                        ],
                    ]
                ],
            ]
        ],
        [
            'title' => 'ابزارها و نگهداری',
            'items' => [
                [
                    'icon' => 'database',
                    'text' => 'پشتیبان‌گیری',
                    'route' => '#',
                    'active' => request()->routeIs('admin.backup.*'),
                    'submenu' => [
                        [
                            'icon' => 'download',
                            'text' => 'ایجاد پشتیبان',
                            'route' => 'admin.backup.create',
                            'active' => request()->routeIs('admin.backup.create')
                        ],
                        [
                            'icon' => 'list',
                            'text' => 'لیست پشتیبان‌ها',
                            'route' => 'admin.backup.index',
                            'active' => request()->routeIs('admin.backup.index')
                        ],
                    ]
                ],
                [
                    'icon' => 'list',
                    'text' => 'لاگ‌ها',
                    'route' => 'admin.logs.index',
                    'active' => request()->routeIs('admin.logs.*')
                ],
                [
                    'icon' => 'bell',
                    'text' => 'اعلان‌ها',
                    'route' => '#',
                    'active' => request()->routeIs('admin.notifications.*'),
                    'submenu' => [
                        [
                            'icon' => 'bell',
                            'text' => 'لیست اعلان‌ها',
                            'route' => 'admin.admin.notifications.index',
                            'active' => request()->routeIs('admin.admin.notifications.index')
                        ],
                        [
                            'icon' => 'cog',
                            'text' => 'تنظیمات اعلان‌ها',
                            'route' => 'admin.admin.notifications.settings',
                            'active' => request()->routeIs('admin.admin.notifications.settings')
                        ],
                    ]
                ],
                [
                    'icon' => 'tools',
                    'text' => 'ابزارها',
                    'route' => '#',
                    'active' => request()->routeIs('admin.tools.*'),
                    'submenu' => [
                        [
                            'icon' => 'sync',
                            'text' => 'بهینه‌سازی دیتابیس',
                            'route' => 'admin.tools.optimize-database',
                            'active' => request()->routeIs('admin.tools.optimize-database')
                        ],
                        [
                            'icon' => 'trash',
                            'text' => 'پاکسازی کش',
                            'route' => 'admin.tools.clear-cache',
                            'active' => request()->routeIs('admin.tools.clear-cache'),
                            'form' => true,
                            'method' => 'POST'
                        ],
                        [
                            'icon' => 'search',
                            'text' => 'جستجوی پیشرفته',
                            'route' => 'admin.tools.advanced-search',
                            'active' => request()->routeIs('admin.tools.advanced-search')
                        ],
                    ]
                ],
                [
                    'icon' => 'database',
                    'text' => 'ابزارهای دیتابیس',
                    'route' => 'admin.tools.database',
                    'active' => request()->routeIs('admin.tools.database')
                ],
            ]
        ],
    ];
@endphp

<div class="sidebar" id="sidebar">
    <div class="sidebar-brand">
        <h3><i class="fas fa-user-shield"></i> پنل مدیریت</h3>
        <div class="sidebar-stats">
            <div class="stat-item">
                <span class="stat-number">{{ $sidebarStats['total_locations'] }}</span>
                <span class="stat-label">مکان</span>
            </div>
            <div class="stat-item">
                <span class="stat-number">{{ $sidebarStats['total_users'] }}</span>
                <span class="stat-label">کاربر</span>
            </div>
            <div class="stat-item">
                <span class="stat-number">{{ $sidebarStats['pending_ratings'] }}</span>
                <span class="stat-label">در انتظار</span>
            </div>
        </div>
    </div>

    @foreach($menuStructure as $section)
        <hr class="sidebar-divider">
        @if(isset($section['title']))
            <div class="sidebar-heading">
                {{ $section['title'] }}
            </div>
        @endif
        <ul class="nav flex-column">
            @foreach($section['items'] as $item)
                @if(isset($item['submenu']))
                    <li class="nav-item">
                        <a href="{{ $item['route'] === '#' ? '#' : route($item['route']) }}"
                           class="nav-link dropdown-toggle {{ $item['active'] ? 'active' : '' }}"
                           data-toggle="collapse"
                           data-target="#submenu_{{ Str::slug($item['text']) }}"
                           aria-expanded="{{ $item['active'] ? 'true' : 'false' }}">
                            <div class="nav-link-content">
                                <div class="nav-link-main">
                                    <i class="fas fa-{{ $item['icon'] }}"></i>
                                    <span>{{ $item['text'] }}</span>
                                </div>
                                <div class="nav-link-badges">
                                    @if(isset($item['badge']))
                                        <span class="badge badge-{{ $item['badge']['type'] }}">{{ $item['badge']['text'] }}</span>
                                    @endif
                                    <i class="fas fa-chevron-down submenu-arrow"></i>
                                </div>
                            </div>
                        </a>
                        <ul class="submenu collapse {{ $item['active'] ? 'show' : '' }}"
                            id="submenu_{{ Str::slug($item['text']) }}"
                            data-parent="#sidebar">
                            @foreach($item['submenu'] as $sub)
                                <li class="nav-item">
                                    @if(isset($sub['form']) && $sub['form'])
                                        <form action="{{ route($sub['route']) }}" method="{{ $sub['method'] }}" class="d-inline">
                                            @csrf
                                            <button type="submit" class="nav-link {{ $sub['active'] ? 'active' : '' }} w-100 text-right">
                                                <div class="nav-link-content">
                                                    <div class="nav-link-main">
                                                        <i class="fas fa-{{ $sub['icon'] }}"></i>
                                                        <span>{{ $sub['text'] }}</span>
                                                    </div>
                                                    @if(isset($sub['badge']))
                                                        <span class="badge badge-{{ $sub['badge']['type'] }}">{{ $sub['badge']['text'] }}</span>
                                                    @endif
                                                </div>
                                            </button>
                                        </form>
                                    @else
                                        <a href="{{ route($sub['route']) }}"
                                           class="nav-link {{ $sub['active'] ? 'active' : '' }}">
                                            <div class="nav-link-content">
                                                <div class="nav-link-main">
                                                    <i class="fas fa-{{ $sub['icon'] }}"></i>
                                                    <span>{{ $sub['text'] }}</span>
                                                </div>
                                                @if(isset($sub['badge']))
                                                    <span class="badge badge-{{ $sub['badge']['type'] }}">{{ $sub['badge']['text'] }}</span>
                                                @endif
                                            </div>
                                        </a>
                                    @endif
                                </li>
                            @endforeach
                        </ul>
                    </li>
                @else
                    <li class="nav-item">
                        <a href="{{ $item['route'] === '#' ? '#' : route($item['route']) }}"
                           class="nav-link {{ $item['active'] ? 'active' : '' }}">
                            <div class="nav-link-content">
                                <div class="nav-link-main">
                                    <i class="fas fa-{{ $item['icon'] }}"></i>
                                    <span>{{ $item['text'] }}</span>
                                </div>
                                @if(isset($item['badge']))
                                    <span class="badge badge-{{ $item['badge']['type'] }}">{{ $item['badge']['text'] }}</span>
                                @endif
                            </div>
                        </a>
                    </li>
                @endif
            @endforeach
        </ul>
    @endforeach

    <!-- System Status Footer -->
    <div class="sidebar-footer">
        <div class="system-status">
            <div class="status-item">
                <i class="fas fa-circle text-success"></i>
                <span>سیستم آنلاین</span>
            </div>
            <div class="status-item">
                <i class="fas fa-database"></i>
                <span>دیتابیس متصل</span>
            </div>
            <div class="status-item">
                <i class="fas fa-clock"></i>
                <span id="current-time">{{ now()->format('H:i') }}</span>
            </div>
        </div>
    </div>
</div>

<style>
/* Modern Professional Sidebar Design */
.sidebar {
    width: 300px;
    height: 100vh;
    position: fixed;
    top: 0;
    right: 0;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: #ffffff;
    overflow-y: auto;
    transition: all 0.4s cubic-bezier(0.25, 0.8, 0.25, 1);
    z-index: 1000;
    box-shadow: -5px 0 25px rgba(0, 0, 0, 0.1);
    backdrop-filter: blur(10px);
}

.sidebar-brand {
    padding: 2rem 1.5rem;
    text-align: center;
    border-bottom: 1px solid rgba(255, 255, 255, 0.15);
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(10px);
}

.sidebar-brand h3 {
    margin: 0;
    font-size: 1.6rem;
    color: #ffffff;
    font-weight: 700;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.3);
}

.sidebar-brand i {
    margin-left: 0.5rem;
    font-size: 1.4rem;
    color: #ffd700;
}

.sidebar-stats {
    display: flex;
    justify-content: space-around;
    margin-top: 1rem;
    padding-top: 1rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
}

.stat-item {
    text-align: center;
    flex: 1;
}

.stat-number {
    display: block;
    font-size: 1.2rem;
    font-weight: 700;
    color: #ffd700;
    text-shadow: 0 1px 3px rgba(0, 0, 0, 0.3);
}

.stat-label {
    display: block;
    font-size: 0.7rem;
    color: rgba(255, 255, 255, 0.7);
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-top: 0.2rem;
}

.sidebar-heading {
    padding: 1rem 1.5rem 0.5rem;
    font-size: 0.8rem;
    text-transform: uppercase;
    color: rgba(255, 255, 255, 0.7);
    font-weight: 600;
    letter-spacing: 1px;
    margin-top: 0.5rem;
}

.sidebar-divider {
    margin: 0.5rem 1.5rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
}

.nav-item {
    position: relative;
    margin: 0.2rem 0.75rem;
}

.nav-link {
    padding: 0;
    color: rgba(255, 255, 255, 0.9);
    display: block;
    transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
    border-radius: 12px;
    text-decoration: none;
    position: relative;
    overflow: hidden;
}

.nav-link-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 0.875rem 1rem;
    position: relative;
    z-index: 2;
}

.nav-link-main {
    display: flex;
    align-items: center;
    flex: 1;
}

.nav-link-badges {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.nav-link::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(135deg, rgba(255, 255, 255, 0.1) 0%, rgba(255, 255, 255, 0.05) 100%);
    opacity: 0;
    transition: opacity 0.3s ease;
    border-radius: 12px;
}

.nav-link:hover::before {
    opacity: 1;
}

.nav-link:hover {
    color: #ffffff;
    transform: translateX(-3px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
}

.nav-link.active {
    background: linear-gradient(135deg, rgba(255, 255, 255, 0.2) 0%, rgba(255, 255, 255, 0.1) 100%);
    color: #ffffff;
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.2);
    border: 1px solid rgba(255, 255, 255, 0.2);
}

.nav-link.active::before {
    opacity: 1;
}

.nav-link i {
    width: 24px;
    margin-left: 0.75rem;
    text-align: center;
    font-size: 1.1rem;
}

/* Badge Styles */
.badge {
    padding: 0.25rem 0.5rem;
    border-radius: 12px;
    font-size: 0.7rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
}

.badge-primary {
    background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
    color: #ffffff;
}

.badge-success {
    background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);
    color: #ffffff;
}

.badge-warning {
    background: linear-gradient(135deg, #fa709a 0%, #fee140 100%);
    color: #ffffff;
}

.badge-danger {
    background: linear-gradient(135deg, #ff6b6b 0%, #ee5a24 100%);
    color: #ffffff;
}

.badge-info {
    background: linear-gradient(135deg, #74b9ff 0%, #0984e3 100%);
    color: #ffffff;
}

.badge-secondary {
    background: linear-gradient(135deg, #a8a8a8 0%, #7f8c8d 100%);
    color: #ffffff;
}

/* Submenu Styles */
.submenu {
    background: rgba(0, 0, 0, 0.15);
    margin: 0.5rem 0;
    border-radius: 12px;
    padding: 0;
    backdrop-filter: blur(5px);
    border: 1px solid rgba(255, 255, 255, 0.1);
    max-height: 0;
    overflow: hidden;
    transition: max-height 0.4s cubic-bezier(0.25, 0.8, 0.25, 1),
                padding 0.4s ease,
                opacity 0.3s ease;
    opacity: 0;
}

.submenu.show {
    padding: 0.5rem 0;
    opacity: 1;
}

/* Force visibility for debugging */
.submenu.collapse {
    display: block !important;
}

.submenu .nav-item {
    margin: 0.1rem 0.5rem;
}

.submenu .nav-link {
    padding: 0;
    border-radius: 8px;
}

.submenu .nav-link-content {
    padding: 0.625rem 1rem 0.625rem 2.5rem;
}

.submenu .nav-link i {
    font-size: 0.9rem;
    width: 20px;
}

.submenu-arrow {
    transition: transform 0.4s cubic-bezier(0.25, 0.8, 0.25, 1);
    font-size: 0.8rem;
    transform-origin: center;
}

.nav-link[aria-expanded="true"] .submenu-arrow {
    transform: rotate(180deg);
}

/* Enhanced hover effects for menu items with submenus */
.nav-link.dropdown-toggle:hover .submenu-arrow {
    transform: scale(1.1);
}

.nav-link.dropdown-toggle[aria-expanded="true"]:hover .submenu-arrow {
    transform: rotate(180deg) scale(1.1);
}

/* Custom Scrollbar */
.sidebar::-webkit-scrollbar {
    width: 8px;
}

.sidebar::-webkit-scrollbar-track {
    background: rgba(255, 255, 255, 0.05);
    border-radius: 4px;
}

.sidebar::-webkit-scrollbar-thumb {
    background: linear-gradient(135deg, rgba(255, 255, 255, 0.3) 0%, rgba(255, 255, 255, 0.1) 100%);
    border-radius: 4px;
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.sidebar::-webkit-scrollbar-thumb:hover {
    background: linear-gradient(135deg, rgba(255, 255, 255, 0.4) 0%, rgba(255, 255, 255, 0.2) 100%);
}

/* Animations */
.nav-link, .submenu, .badge {
    transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
}

.submenu.collapse {
    transition: all 0.4s cubic-bezier(0.25, 0.8, 0.25, 1);
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .sidebar {
        width: 100%;
        transform: translateX(100%);
    }

    .sidebar.show {
        transform: translateX(0);
    }

    .nav-link-content {
        padding: 1rem 1.5rem;
    }

    .sidebar-brand {
        padding: 1.5rem;
    }
}

/* Dark mode support */
@media (prefers-color-scheme: dark) {
    .sidebar {
        background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
    }
}

/* Hover effects for better UX */
.nav-item:hover .badge {
    transform: scale(1.05);
}

/* Focus states for accessibility */
.nav-link:focus {
    outline: 2px solid rgba(255, 255, 255, 0.5);
    outline-offset: 2px;
}

/* Sidebar Footer */
.sidebar-footer {
    margin-top: auto;
    padding: 1rem 1.5rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
    background: rgba(0, 0, 0, 0.1);
}

.system-status {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.status-item {
    display: flex;
    align-items: center;
    font-size: 0.75rem;
    color: rgba(255, 255, 255, 0.8);
}

.status-item i {
    width: 16px;
    margin-left: 0.5rem;
    font-size: 0.7rem;
}

.text-success {
    color: #28a745 !important;
}

/* Real-time clock animation */
#current-time {
    font-weight: 600;
    color: #ffd700;
}
</style>

<script>
// Update time every minute
setInterval(function() {
    const now = new Date();
    const timeString = now.getHours().toString().padStart(2, '0') + ':' +
                      now.getMinutes().toString().padStart(2, '0');
    const timeElement = document.getElementById('current-time');
    if (timeElement) {
        timeElement.textContent = timeString;
    }
}, 60000);

// Initialize sidebar functionality
document.addEventListener('DOMContentLoaded', function() {
    // Find all dropdown toggle buttons
    const dropdownToggles = document.querySelectorAll('a[data-toggle="collapse"]');

    dropdownToggles.forEach(function(toggle) {
        toggle.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();

            // Get target submenu
            const targetSelector = this.getAttribute('data-target');
            const targetSubmenu = document.querySelector(targetSelector);

            if (targetSubmenu) {
                // Check if currently open
                const isCurrentlyOpen = targetSubmenu.classList.contains('show');

                // Close all other submenus first
                document.querySelectorAll('.submenu').forEach(function(submenu) {
                    if (submenu !== targetSubmenu) {
                        submenu.classList.remove('show');
                        submenu.style.maxHeight = '0px';
                        submenu.style.opacity = '0';
                        // Update aria-expanded for other buttons
                        const otherButton = document.querySelector('[data-target="#' + submenu.id + '"]');
                        if (otherButton) {
                            otherButton.setAttribute('aria-expanded', 'false');
                        }
                    }
                });

                // Toggle current submenu
                if (isCurrentlyOpen) {
                    // Close it
                    targetSubmenu.classList.remove('show');
                    targetSubmenu.style.maxHeight = '0px';
                    targetSubmenu.style.opacity = '0';
                    this.setAttribute('aria-expanded', 'false');
                } else {
                    // Open it
                    targetSubmenu.classList.add('show');
                    targetSubmenu.style.maxHeight = targetSubmenu.scrollHeight + 'px';
                    targetSubmenu.style.opacity = '1';
                    this.setAttribute('aria-expanded', 'true');
                }
            }
        });
    });

    // Initialize submenus that should be open by default
    document.querySelectorAll('.submenu.show').forEach(function(submenu) {
        submenu.style.maxHeight = submenu.scrollHeight + 'px';
        submenu.style.opacity = '1';
    });
});
</script>