/**
 * Map Fallback System
 * 
 * This script provides fallback functionality when the map dependencies fail to load.
 * It creates mock objects that prevent JavaScript errors and ensures the category filter
 * continues to work properly.
 */

// Check if Leaflet loaded properly
if (typeof L === 'undefined') {
    console.warn('Leaflet library failed to load - creating fallback objects');
    
    // Create a global L object with required functions
    window.L = {
        // Minimal implementation to prevent errors
        latLng: function(lat, lng) {
            return { lat: lat, lng: lng };
        },
        marker: function() {
            return {
                addTo: function() { return this; },
                setLatLng: function() { return this; },
                bindPopup: function() { return this; }
            };
        },
        map: function() {
            return {
                setView: function() { return this; },
                addLayer: function() { return this; },
                removeLayer: function() { return this; },
                fitBounds: function() { return this; }
            };
        }
    };
}

// Create a minimal map instance if the initialization failed
if (!window.mapInstance) {
    console.warn('Map initialization failed - creating mock map instance');
    
    window.mapInstance = {
        // Basic methods that the category filter might call
        updateLocations: function(locations) {
            console.log('Mock map received locations update:', locations.length);
            // Update the list view even if map fails
            if (typeof updateLocationsList === 'function') {
                updateLocationsList(locations);
            }
            return this;
        },
        clearMarkers: function() {
            console.log('Mock map clearing markers');
            return this;
        },
        panTo: function() {
            console.log('Mock map pan attempt');
            return this;
        },
        fitBounds: function() {
            console.log('Mock map fit bounds attempt');
            return this;
        },
        showLoading: function() {
            return this;
        },
        hideLoading: function() {
            return this;
        }
    };
}

// Ensure the category filter doesn't break due to map errors
document.addEventListener('DOMContentLoaded', function() {
    console.log('Map fallback system initialized');
    
    // Find any map error messages and display them properly
    const mapContainer = document.getElementById('map');
    if (mapContainer) {
        // If the map failed to load, show a helpful message
        if (typeof L === 'undefined' || !window.mapInstance.isValid) {
            mapContainer.innerHTML = `
                <div class="map-error-container">
                    <div class="map-error-message">
                        <h3>نقشه در حال حاضر در دسترس نیست</h3>
                        <p>لطفاً از فیلترها برای مشاهده لیست مکان‌ها استفاده کنید.</p>
                    </div>
                </div>
            `;
            
            // Add some basic styling to the error message
            const style = document.createElement('style');
            style.textContent = `
                .map-error-container {
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    height: 100%;
                    background-color: #f5f5f5;
                    border-radius: 8px;
                }
                .map-error-message {
                    text-align: center;
                    padding: 20px;
                    background-color: white;
                    border-radius: 8px;
                    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
                    max-width: 80%;
                }
            `;
            document.head.appendChild(style);
        }
    }
    
    // Add fallback for geo-filter errors
    if (typeof window.geoFilterHandler === 'undefined' || 
        typeof window.categoryFilter === 'undefined') {
        
        console.log('Creating fallback filter handlers');
        
        // Create minimal category filter if it doesn't exist
        if (typeof window.categoryFilter === 'undefined') {
            window.categoryFilter = {
                updateCategoryCounts: function(counts) {
                    console.log('Fallback updating category counts', counts);
                },
                setActiveCategory: function(id) {
                    console.log('Fallback setting active category', id);
                }
            };
        }
        
        // Create minimal geo filter if it doesn't exist
        if (typeof window.geoFilterHandler === 'undefined') {
            window.geoFilterHandler = {
                applyFilter: function(categoryId, includeChildren) {
                    console.log('Fallback applying filter', categoryId, includeChildren);
                    
                    // Make API call directly
                    const params = new URLSearchParams();
                    if (categoryId) {
                        params.append('category_id', categoryId);
                        if (includeChildren) {
                            params.append('include_children', 1);
                        }
                    }
                    params.append('is_active', 1);
                    
                    fetch(`/api/locations?${params.toString()}`)
                        .then(response => response.json())
                        .then(data => {
                            const locations = data.data || [];
                            console.log(`Fallback loaded ${locations.length} locations`);
                            
                            // Update list view
                            if (typeof updateLocationsList === 'function') {
                                updateLocationsList(locations);
                            }
                        })
                        .catch(error => {
                            console.error('Error in fallback filter:', error);
                        });
                }
            };
        }
        
        // Connect category pills to the fallback system
        const categoryPills = document.querySelectorAll('.category-pill, .parent-category');
        categoryPills.forEach(pill => {
            // Remove existing listeners by cloning
            const newPill = pill.cloneNode(true);
            pill.parentNode.replaceChild(newPill, pill);
            
            // Add our fallback listener
            newPill.addEventListener('click', function() {
                const categoryId = this.dataset.categoryId;
                const isParent = this.classList.contains('parent-category');
                
                console.log('Fallback category click', categoryId, isParent);
                
                // Update visual state
                document.querySelectorAll('.category-pill, .parent-category').forEach(p => {
                    p.classList.remove('active');
                });
                this.classList.add('active');
                
                // Show children if parent
                if (isParent) {
                    const childContainer = this.nextElementSibling;
                    if (childContainer && childContainer.classList.contains('subcategories')) {
                        childContainer.style.display = 'flex';
                    }
                }
                
                // Apply filter
                if (window.geoFilterHandler) {
                    window.geoFilterHandler.applyFilter(categoryId, isParent);
                }
            });
        });
    }
}); 