/**
 * MapApiAdapter Class
 *
 * Handles communication with the backend API for locations and categories.
 * Abstracts API calls and data transformations to provide clean data to the controller.
 */
class MapApiAdapter {
    /**
     * Constructor for MapApiAdapter
     * @param {Object} options - Configuration options
     * @param {string} options.baseUrl - Base URL for API calls (defaults to current origin)
     * @param {Object} options.headers - Default headers for API calls
     * @param {boolean} options.debug - Enable debug logging
     */
    constructor(options = {}) {
        this.options = {
            baseUrl: window.location.origin,
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
            },
            debug: false,
            ...options
        };

        this.endpoints = {
            categories: '/api/categories',
            locations: '/api/locations',
            location: (id) => `/api/location/${id}`
        };

        this.log('MapApiAdapter initialized');
    }

    /**
     * Fetch categories from the API
     * @param {Object} params - Parameters for the request
     * @returns {Promise<Array>} List of categories
     */
    async getCategories(params = {}) {
        try {
            const url = this.buildUrl(this.endpoints.categories, params);
            this.log('Fetching categories', url);

            const response = await fetch(url, {
                method: 'GET',
                headers: this.options.headers
            });

            if (!response.ok) {
                throw new Error(`Error fetching categories: ${response.status}`);
            }

            const data = await response.json();
            this.log('Categories fetched', data);

            return data.data || [];
        } catch (error) {
            console.error('Error fetching categories:', error);
            throw error;
        }
    }

    /**
     * Fetch locations from the API
     * @param {Object} params - Parameters for the request
     * @param {number|string} params.category_id - Category ID to filter by
     * @param {boolean} params.include_children - Whether to include child categories
     * @param {string} params.search - Search query
     * @returns {Promise<Object>} Locations data including meta information
     */
    async getLocations(params = {}) {
        try {
            const url = this.buildUrl(this.endpoints.locations, params);
            this.log('Fetching locations', url);

            const response = await fetch(url, {
                method: 'GET',
                headers: this.options.headers
            });

            if (!response.ok) {
                throw new Error(`Error fetching locations: ${response.status}`);
            }

            const data = await response.json();
            this.log('Locations fetched', data);

            return {
                locations: data.data || [],
                meta: data.meta || {}
            };
        } catch (error) {
            console.error('Error fetching locations:', error);
            throw error;
        }
    }

    /**
     * Fetch a single location by ID
     * @param {number|string} id - Location ID
     * @returns {Promise<Object>} Location data
     */
    async getLocation(id) {
        if (!id) {
            throw new Error('Location ID is required');
        }

        try {
            const url = this.buildUrl(this.endpoints.location(id));
            this.log(`Fetching location ${id}`, url);

            const response = await fetch(url, {
                method: 'GET',
                headers: this.options.headers
            });

            if (!response.ok) {
                throw new Error(`Error fetching location: ${response.status}`);
            }

            const data = await response.json();
            this.log('Location fetched', data);

            return data.data || null;
        } catch (error) {
            console.error(`Error fetching location ${id}:`, error);
            throw error;
        }
    }

    /**
     * Search locations by query string
     * @param {string} query - Search query
     * @returns {Promise<Object>} Search results
     */
    async searchLocations(query) {
        return this.getLocations({ search: query });
    }

    /**
     * Get locations for a specific category
     * @param {number|string} categoryId - Category ID
     * @param {boolean} includeChildren - Whether to include child categories
     * @returns {Promise<Object>} Locations for the category
     */
    async getCategoryLocations(categoryId, includeChildren = true) {
        return this.getLocations({
            category: categoryId,
            include_children: includeChildren ? 1 : 0
        });
    }

    /**
     * Get locations for child categories of a parent category
     * @param {number|string} parentCategoryId - Parent category ID
     * @param {boolean} isActive - Whether to include only active locations
     * @returns {Promise<Object>} Locations for the child categories
     */
    async getChildCategoryLocations(parentCategoryId, isActive = true) {
        return this.getLocations({
            parent_category_id: parentCategoryId,
            is_active: isActive ? 1 : 0,
            count_by_category: 1
        });
    }

    /**
     * Build a URL with query parameters
     * @param {string} endpoint - API endpoint
     * @param {Object} params - Query parameters
     * @returns {string} Full URL with query parameters
     */
    buildUrl(endpoint, params = {}) {
        const url = new URL(this.options.baseUrl + endpoint);

        // Add query parameters
        Object.entries(params).forEach(([key, value]) => {
            if (value !== undefined && value !== null) {
                url.searchParams.append(key, value);
            }
        });

        return url.toString();
    }

    /**
     * Log debug messages if debug is enabled
     * @param {string} message - Debug message
     * @param {*} data - Optional data to log
     */
    log(message, data) {
        if (this.options.debug) {
            if (data !== undefined) {
                console.log(`[MapApiAdapter] ${message}`, data);
            } else {
                console.log(`[MapApiAdapter] ${message}`);
            }
        }
    }
}