/**
 * Category Filter Component
 *
 * Handles UI interactions for category filtering:
 * - Manages parent/child category relationships
 * - Provides proper expand/collapse functionality
 * - Ensures correct behavior when switching between parent categories
 * - Updates UI based on API responses
 * - Automatically displays child categories when parent is selected
 */

class CategoryFilter {
    constructor(options = {}) {
        // Configuration
        this.options = {
            containerSelector: options.containerSelector || '.category-filter-container',
            parentCategorySelector: options.parentCategorySelector || '.parent-category',
            childCategorySelector: options.childCategorySelector || '.child-category',
            activeCategoryClass: options.activeCategoryClass || 'active',
            expandedParentClass: options.expandedParentClass || 'expanded',
            hiddenClass: options.hiddenClass || 'hidden',
            childrenContainerClass: options.childrenContainerClass || 'children-container',
            categoryCountClass: options.categoryCountClass || 'category-count',
            clearFilterSelector: options.clearFilterSelector || '.clear-filter',
            onCategoryChange: options.onCategoryChange || null
        };

        // State
        this.parentChildMap = {};
        this.expandedParents = new Set();
        this.activeCategory = null;
        this.categoryCountMap = {};

        // Element references
        this.containerEl = document.querySelector(this.options.containerSelector);

        // Initialize component
        this.init();
    }

    /**
     * Initialize the component
     */
    init() {
        if (!this.containerEl) {
            console.error('Category filter container not found:', this.options.containerSelector);
            return;
        }

        this.buildParentChildMap();
        this.attachEventListeners();
        console.log('CategoryFilter initialized');
    }

    /**
     * Build mapping of parent categories to child categories
     */
    buildParentChildMap() {
        const parentEls = this.containerEl.querySelectorAll(this.options.parentCategorySelector);

        parentEls.forEach(parentEl => {
            const parentId = parentEl.dataset.categoryId;
            const childrenContainer = parentEl.nextElementSibling;

            if (childrenContainer && childrenContainer.classList.contains(this.options.childrenContainerClass)) {
                const childEls = childrenContainer.querySelectorAll(this.options.childCategorySelector);
                this.parentChildMap[parentId] = Array.from(childEls).map(el => el.dataset.categoryId);

                // Store reference to the children container for this parent
                parentEl.childrenContainer = childrenContainer;
            }
        });

        console.log('Parent-child map built:', this.parentChildMap);
    }

    /**
     * Attach event listeners
     */
    attachEventListeners() {
        // Parent category click handler
        this.containerEl.querySelectorAll(this.options.parentCategorySelector).forEach(el => {
            el.addEventListener('click', (e) => this.handleCategoryClick(e, true));
        });

        // Child category click handler
        this.containerEl.querySelectorAll(this.options.childCategorySelector).forEach(el => {
            el.addEventListener('click', (e) => this.handleCategoryClick(e, false));
        });

        // Clear filter button
        const clearFilterBtn = document.querySelector(this.options.clearFilterSelector);
        if (clearFilterBtn) {
            clearFilterBtn.addEventListener('click', () => this.clearFilters());
        }

        console.log('Event listeners attached');
    }

    /**
     * Handle category click
     * @param {Event} e - Click event
     * @param {boolean} isParent - Whether the clicked category is a parent
     */
    handleCategoryClick(e, isParent) {
        const categoryEl = e.currentTarget;
        const categoryId = categoryEl.dataset.categoryId;

        console.log(`Category clicked: ${categoryId} (${isParent ? 'parent' : 'child'})`);

        // Check if clicking on expand icon, category meta, or category count
        const isExpandIconClick = e.target.closest('.expand-icon') || e.target.classList.contains('expand-icon');
        const isCategoryMetaClick = e.target.closest('.category-meta') || e.target.classList.contains('category-meta');
        const isCategoryCountClick = e.target.closest('.category-count') || e.target.classList.contains('category-count');

        console.log('Click details:', {
            isExpandIconClick,
            isCategoryMetaClick,
            isCategoryCountClick,
            targetClass: e.target.className,
            targetTagName: e.target.tagName
        });

        // اگر روی دسته‌ای کلیک شده که از قبل فعال است، باید آن را غیرفعال کنیم
        const isAlreadyActive = this.activeCategory === categoryId;

        if (isParent) {
            const childrenContainer = categoryEl.childrenContainer;
            const hasChildren = childrenContainer && childrenContainer.querySelector('.child-category');

            if ((isExpandIconClick || isCategoryMetaClick || isCategoryCountClick) && hasChildren) {
                // Just toggle expansion for icon/meta/count clicks
                e.preventDefault();
                e.stopPropagation();

                console.log(`Toggling expansion for parent ${categoryId}`);
                if (this.expandedParents.has(categoryId)) {
                    this.collapseParent(categoryId, categoryEl);
                } else {
                    this.expandParent(categoryId, categoryEl);
                }
                return; // Don't apply filter, just toggle expansion
            }

            // For content clicks, handle selection
            if (isAlreadyActive) {
                console.log(`Category ${categoryId} is already active - deactivating`);
                this.clearFilters();
                return;
            }

            // Always collapse all other parents when clicking a parent category
            this.collapseAllParents();

            // Apply filter for parent category with include_children=true
            this.setActiveCategory(categoryId, categoryEl);

            // Also expand this parent if it has children
            if (hasChildren) {
                this.expandParent(categoryId, categoryEl);
            }

            if (this.options.onCategoryChange) {
                // Send explicit flag indicating this is a parent category
                console.log(`Triggering onCategoryChange for parent category ${categoryId} with include_children=true and isParent=true`);
                this.options.onCategoryChange(categoryId, {
                    include_children: true,
                    resetPrevious: true,
                    isParent: true  // Explicit flag to indicate parent category
                });
            }
        } else {
            // Child category clicked
            if (isAlreadyActive) {
                console.log(`Category ${categoryId} is already active - deactivating`);
                this.clearFilters();
                return;
            }

            // Get parent of this child category
            const parentId = this.getParentForChild(categoryId);

            // Make sure the parent is expanded
            if (parentId) {
                this.expandParent(parentId);
            }

            // Child category was clicked - set active and apply filter
            this.setActiveCategory(categoryId, categoryEl);

            if (this.options.onCategoryChange) {
                // Send explicit flag indicating this is NOT a parent category
                console.log(`Triggering onCategoryChange for child category ${categoryId} with include_children=false and isParent=false`);
                this.options.onCategoryChange(categoryId, {
                    include_children: false,
                    resetPrevious: false,
                    parentId: parentId,
                    isParent: false  // Explicit flag to indicate child category
                });
            }
        }
    }

    /**
     * Toggle parent category expanded state
     * @param {string} parentId - Parent category ID
     * @param {HTMLElement} parentEl - Parent category element
     */
    toggleParentExpanded(parentId, parentEl) {
        if (this.expandedParents.has(parentId)) {
            this.collapseParent(parentId, parentEl);
        } else {
            this.expandParent(parentId, parentEl);
        }
    }

    /**
     * Expand parent category to show children
     * @param {string} parentId - Parent category ID
     * @param {HTMLElement} parentEl - Parent category element
     */
    expandParent(parentId, parentEl) {
        if (!parentEl && parentId) {
            parentEl = this.containerEl.querySelector(`${this.options.parentCategorySelector}[data-category-id="${parentId}"]`);
        }

        if (parentEl) {
            parentEl.classList.add(this.options.expandedParentClass);
            const childrenContainer = parentEl.childrenContainer;

            if (childrenContainer) {
                childrenContainer.classList.remove(this.options.hiddenClass);
            }

            this.expandedParents.add(parentId);
            console.log(`Expanded parent: ${parentId}`);
        }
    }

    /**
     * Collapse parent category to hide children
     * @param {string} parentId - Parent category ID
     * @param {HTMLElement} parentEl - Parent category element
     */
    collapseParent(parentId, parentEl) {
        if (!parentEl && parentId) {
            parentEl = this.containerEl.querySelector(`${this.options.parentCategorySelector}[data-category-id="${parentId}"]`);
        }

        if (parentEl) {
            parentEl.classList.remove(this.options.expandedParentClass);
            const childrenContainer = parentEl.childrenContainer;

            if (childrenContainer) {
                childrenContainer.classList.add(this.options.hiddenClass);
            }

            this.expandedParents.delete(parentId);
            console.log(`Collapsed parent: ${parentId}`);
        }
    }

    /**
     * Collapse all parent categories
     */
    collapseAllParents() {
        const expandedParentIds = [...this.expandedParents];

        expandedParentIds.forEach(parentId => {
            this.collapseParent(parentId);
        });

        console.log('Collapsed all parents');
    }

    /**
     * Set active category
     * @param {string} categoryId - Category ID
     * @param {HTMLElement} categoryEl - Category element
     */
    setActiveCategory(categoryId, categoryEl) {
        // Clear previous active category
        const activeEls = this.containerEl.querySelectorAll(`.${this.options.activeCategoryClass}`);
        activeEls.forEach(el => el.classList.remove(this.options.activeCategoryClass));

        // Set new active category
        if (!categoryEl && categoryId) {
            categoryEl = this.containerEl.querySelector(`[data-category-id="${categoryId}"]`);
        }

        if (categoryEl) {
            categoryEl.classList.add(this.options.activeCategoryClass);
        }

        this.activeCategory = categoryId;
        console.log(`Active category set: ${categoryId}`);

        // If it's a parent, make sure it's expanded
        if (this.isParentCategory(categoryId)) {
            this.expandParent(categoryId, categoryEl);
        }
    }

    /**
     * Check if a category is a parent category
     * @param {string} categoryId - Category ID
     * @returns {boolean} True if the category is a parent
     */
    isParentCategory(categoryId) {
        return Object.keys(this.parentChildMap).includes(categoryId);
    }

    /**
     * Get the parent category ID for a child category
     * @param {string} childId - Child category ID
     * @returns {string|null} Parent category ID or null if not found
     */
    getParentForChild(childId) {
        for (const [parentId, children] of Object.entries(this.parentChildMap)) {
            if (children.includes(childId)) {
                return parentId;
            }
        }
        return null;
    }

    /**
     * Clear all filters
     */
    clearFilters() {
        // Clear active category
        const activeEls = this.containerEl.querySelectorAll(`.${this.options.activeCategoryClass}`);
        activeEls.forEach(el => el.classList.remove(this.options.activeCategoryClass));

        // Collapse all parents
        this.collapseAllParents();

        // Reset state
        this.activeCategory = null;

        // Notify controller
        if (this.options.onCategoryChange) {
            this.options.onCategoryChange(null, {
                include_children: true,
                resetPrevious: true
            });
        }

        console.log('Filters cleared');
    }

    /**
     * Update category counts
     * @param {Object} counts - Map of category IDs to counts
     */
    updateCategoryCounts(counts) {
        if (!counts) {
            console.warn('No counts provided to updateCategoryCounts');
            return;
        }

        console.log('Updating category counts', counts);
        this.categoryCountMap = counts;

        // Update all parent categories
        const parentEls = this.containerEl.querySelectorAll(this.options.parentCategorySelector);
        parentEls.forEach(parentEl => {
            const categoryId = parentEl.dataset.categoryId;
            const countEl = parentEl.querySelector(`.${this.options.categoryCountClass}`);

            if (countEl) {
                const count = counts[categoryId] || 0;
                countEl.textContent = count;

                // Add a class for zero counts
                if (count === 0) {
                    countEl.classList.add('zero');
                } else {
                    countEl.classList.remove('zero');
                }
            }
        });

        // Update all child categories
        const childEls = this.containerEl.querySelectorAll(this.options.childCategorySelector);
        childEls.forEach(childEl => {
            const categoryId = childEl.dataset.categoryId;
            const countEl = childEl.querySelector(`.${this.options.categoryCountClass}`);

            if (countEl) {
                const count = counts[categoryId] || 0;
                countEl.textContent = count;

                // Add a class for zero counts
                if (count === 0) {
                    countEl.classList.add('zero');
                } else {
                    countEl.classList.remove('zero');
                }
            }
        });

        console.log('Category counts updated');
    }

    /**
     * Reset all child category selections when switching parent categories
     * Ensures that no previously selected child categories remain active
     * when a new parent category is selected
     */
    resetChildSelections() {
        // Get all child categories with the active class
        const activeChildEls = this.containerEl.querySelectorAll(
            `${this.options.childCategorySelector}.${this.options.activeCategoryClass}`
        );

        // Remove active class from all child categories
        activeChildEls.forEach(el => {
            el.classList.remove(this.options.activeCategoryClass);
        });

        console.log('Reset all child category selections');
    }
}

// Initialize the category filter when included in a page
window.categoryFilter = new CategoryFilter({
    containerSelector: '.category-filter-container',
    onCategoryChange: (categoryId, options) => {
        console.log('Category filter changed:', categoryId, options);

        // Find the map controller
        if (window.mapController) {
            console.log('Found map controller, handling category change with options:', options);

            // Add debugging to ensure isParent flag is correctly passed
            if (options.isParent !== undefined) {
                console.log('isParent flag is set to:', options.isParent);
            }

            // Use the new dedicated method for category changes
            if (window.mapController.handleCategoryFilterChange) {
                console.log('Using handleCategoryFilterChange method with category ID:', categoryId);
                window.mapController.handleCategoryFilterChange(categoryId, options);
            } else {
                // Fallback to old behavior
                console.log('Fallback: using old behavior');
                window.mapController.state.currentCategoryId = categoryId;
                window.mapController.state.includeChildren = options.include_children;
                window.mapController.loadLocations();
            }
        } else {
            console.warn('Map controller not found, cannot process category change');
        }
    }
});