<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Schema;

class Category extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'category_key',
        'category_fa',
        'category_en',
        'emoji',
        'font_icon',
        'color',
        'sort_order',
        'show_in_menu',
        'show_in_filters',
        'parent_id',
        'description',
        'is_active',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'show_in_menu' => 'boolean',
        'show_in_filters' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get the parent category.
     */
    public function parent()
    {
        return $this->belongsTo(Category::class, 'parent_id');
    }

    /**
     * Get the children categories.
     */
    public function children()
    {
        return $this->hasMany(Category::class, 'parent_id');
    }

    /**
     * Get all nested children categories (recursive).
     */
    public function allChildren()
    {
        return $this->children()->with('allChildren');
    }

    /**
     * Get all child category IDs (including nested children)
     *
     * @return array
     */
    public function getAllChildrenIds()
    {
        try {
            // روش بهینه‌تر برای دریافت همه فرزندان
            $childrenIds = [];

            // ابتدا فرزندان مستقیم را دریافت می‌کنیم
            $directChildren = $this->children()->pluck('id')->toArray();
            $childrenIds = array_merge($childrenIds, $directChildren);

            // سپس برای هر فرزند مستقیم، فرزندان آنها را نیز دریافت می‌کنیم
            foreach ($directChildren as $childId) {
                $child = self::find($childId);
                if ($child) {
                    // افزودن فرزندان مستقیم این زیردسته
                    $childrenIds = array_merge($childrenIds, $child->children()->pluck('id')->toArray());
                }
            }

            // Ensure all IDs are valid
            $childrenIds = array_filter($childrenIds, function ($id) {
                return is_numeric($id) && $id > 0;
            });

            return array_unique($childrenIds);
        } catch (\Exception $e) {
            // Log error but return an empty array to prevent breaking the application
            \Log::error('Error in getAllChildrenIds for category ID '.$this->id.': '.$e->getMessage());

            return [];
        }
    }

    /**
     * Get all locations belonging to this category.
     */
    public function locations()
    {
        return $this->hasMany(Location::class, 'category_id');
    }

    /**
     * Get the category templates
     */
    public function templates()
    {
        return $this->hasMany(CategoryTemplate::class);
    }

    /**
     * Get the active template for this category (as relationship)
     */
    public function activeTemplateRelation()
    {
        return $this->hasOne(CategoryTemplate::class)->where('is_active', true)->orderBy('display_order');
    }

    /**
     * Get the active template for this category (as method)
     */
    public function activeTemplate()
    {
        return $this->hasOne(CategoryTemplate::class)->where('is_active', true)->orderBy('display_order')->first();
    }

    /**
     * Get the template view name for this category
     *
     * @return string
     */
    public function getTemplateViewName()
    {
        $template = $this->activeTemplate();

        if ($template && $template->template_key) {
            return 'templates.locations.'.$template->template_key;
        }

        // If no specific template defined, try to use the one based on category_key
        if ($this->category_key) {
            return 'templates.locations.'.$this->category_key;
        }

        // Fall back to default template
        return 'templates.locations.base';
    }

    /**
     * Get the full path attribute (breadcrumb).
     *
     * @return string
     */
    public function getFullPathAttribute()
    {
        $path = $this->category_fa;

        if ($this->parent) {
            $path = $this->parent->full_path.' / '.$path;
        }

        return $path;
    }

    /**
     * Check if category has any locations.
     *
     * @return bool
     */
    public function hasLocations()
    {
        return $this->locations()->count() > 0;
    }

    // Get all categories as a hierarchical tree
    public static function getTree()
    {
        $categories = self::where('parent_id', null)
            ->with('children')
            ->orderBy('sort_order')
            ->get();

        return $categories;
    }

    /**
     * Get all location IDs related to this category, including from child categories
     *
     * @param  bool  $includeChildren  Whether to include locations from child categories
     * @return array Array of location IDs
     */
    public function getLocationIds($includeChildren = true)
    {
        // Get direct location IDs
        $locationIds = $this->locations()->pluck('id')->toArray();

        // Include children's locations if requested
        if ($includeChildren) {
            $childrenIds = $this->children()->pluck('id')->toArray();
            if (! empty($childrenIds)) {
                $childLocations = Location::whereIn('category_id', $childrenIds)->pluck('id')->toArray();
                $locationIds = array_merge($locationIds, $childLocations);
            }
        }

        return $locationIds;
    }

    /**
     * Check if this category or its children have any locations
     *
     * @return bool
     */
    public function hasLocationsIncludingChildren()
    {
        if ($this->hasLocations()) {
            return true;
        }

        // Check children
        foreach ($this->children as $child) {
            if ($child->hasLocations()) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get location count for this category and its children
     *
     * @return int
     */
    public function getTotalLocationCount()
    {
        // Direct locations count
        $count = $this->locations()->count();

        // Add children locations
        foreach ($this->children as $child) {
            $count += $child->locations()->count();
        }

        return $count;
    }

    public function translations()
    {
        return $this->hasMany(CategoryTranslation::class);
    }

    /**
     * دریافت نام ترجمه‌شده دسته‌بندی بر اساس زبان فعال
     * اگر زبان انگلیسی باشد و category_en مقدار داشته باشد، همان را برمی‌گرداند.
     * در غیر این صورت از ترجمه دیتابیسی یا category_fa استفاده می‌کند.
     *
     * @param string $languageCode کد زبان (fa, en, ...)
     * @return string نام ترجمه‌شده
     */
    public function getTranslatedName($languageCode = null)
    {
        if (!$languageCode) {
            $languageCode = app()->getLocale();
        }

        // Check if we have a direct translation column for this language
        $columnName = 'category_' . $languageCode;
        if (Schema::hasColumn('categories', $columnName) && !empty($this->$columnName)) {
            return $this->$columnName;
        }

        // If no direct translation exists, try to get from translations table
        $translation = $this->translations()
            ->where('language_code', $languageCode)
            ->first();

        if ($translation && !empty($translation->name)) {
            return $translation->name;
        }

        // If no translation found, fallback to English
        if (!empty($this->category_en)) {
            return $this->category_en;
        }

        // If no English translation, fallback to Persian
        return $this->category_fa;
    }

    /**
     * دریافت توضیحات ترجمه‌شده دسته‌بندی بر اساس زبان فعال
     * اگر زبان انگلیسی باشد و description_en مقدار داشته باشد، همان را برمی‌گرداند.
     * در غیر این صورت از ترجمه دیتابیسی یا description_fa استفاده می‌کند.
     *
     * @param string $languageCode کد زبان (fa, en, ...)
     * @return string توضیحات ترجمه‌شده
     */
    public function getTranslatedDescription($languageCode = null)
    {
        if (!$languageCode) {
            $languageCode = app()->getLocale();
        }

        // Check if we have a direct translation column for this language
        $columnName = 'description_' . $languageCode;
        if (Schema::hasColumn('categories', $columnName) && !empty($this->$columnName)) {
            return $this->$columnName;
        }

        // If no direct translation exists, try to get from translations table
        $translation = $this->translations()
            ->where('language_code', $languageCode)
            ->first();

        if ($translation && !empty($translation->description)) {
            return $translation->description;
        }

        // If no translation found, fallback to English
        if (!empty($this->description_en)) {
            return $this->description_en;
        }

        // If no English translation, fallback to Persian
        return $this->description_fa;
    }
}
