<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ButtonSetting extends Model
{
    use HasFactory;

    protected $table = 'button_settings';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'button_key',
        'button_text',
        'button_url',
        'button_icon',
        'button_class',
        'is_visible',
        'show_for_guests',
        'show_for_users',
        'sort_order',
        'language_code',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'is_visible' => 'boolean',
        'show_for_guests' => 'boolean',
        'show_for_users' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get the language.
     */
    public function language()
    {
        return $this->belongsTo(Language::class, 'language_code', 'code');
    }

    /**
     * Scope a query to only include visible buttons.
     */
    public function scopeVisible($query)
    {
        return $query->where('is_visible', true);
    }

    /**
     * Scope a query to filter by language.
     */
    public function scopeForLanguage($query, $languageCode)
    {
        return $query->where('language_code', $languageCode);
    }

    /**
     * Scope a query to filter by user type.
     */
    public function scopeForUserType($query, $isGuest = true)
    {
        if ($isGuest) {
            return $query->where('show_for_guests', true);
        } else {
            return $query->where('show_for_users', true);
        }
    }

    /**
     * Get buttons for specific user type and language.
     */
    public static function getButtonsForUser($languageCode = 'fa', $isGuest = true)
    {
        return self::visible()
            ->forLanguage($languageCode)
            ->forUserType($isGuest)
            ->orderBy('sort_order')
            ->get();
    }

    /**
     * Get button by key and language.
     */
    public static function getButton($key, $languageCode = 'fa')
    {
        return self::visible()
            ->forLanguage($languageCode)
            ->where('button_key', $key)
            ->first();
    }

    /**
     * Update button settings.
     */
    public static function updateButton($key, $data, $languageCode = 'fa')
    {
        return self::updateOrCreate(
            [
                'button_key' => $key,
                'language_code' => $languageCode,
            ],
            $data
        );
    }

    /**
     * Get full URL with domain if needed.
     */
    public function getFullUrlAttribute()
    {
        if (filter_var($this->button_url, FILTER_VALIDATE_URL)) {
            return $this->button_url;
        }
        
        return url($this->button_url);
    }

    /**
     * Get button HTML class.
     */
    public function getButtonClassAttribute($value)
    {
        return $value ?: 'btn btn-primary';
    }
}
