<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class SanitizeInput
{
    protected $xssPatterns = [
        '/<script\b[^>]*>(.*?)<\/script>/is',
        '/<iframe\b[^>]*>(.*?)<\/iframe>/is',
        '/<object\b[^>]*>(.*?)<\/object>/is',
        '/<embed\b[^>]*>(.*?)<\/embed>/is',
        '/<applet\b[^>]*>(.*?)<\/applet>/is',
        '/<form\b[^>]*>(.*?)<\/form>/is',
        '/<input\b[^>]*>(.*?)<\/input>/is',
        '/<textarea\b[^>]*>(.*?)<\/textarea>/is',
        '/<select\b[^>]*>(.*?)<\/select>/is',
        '/<button\b[^>]*>(.*?)<\/button>/is',
        '/<style\b[^>]*>(.*?)<\/style>/is',
        '/<link\b[^>]*>(.*?)<\/link>/is',
        '/<meta\b[^>]*>(.*?)<\/meta>/is',
        '/<base\b[^>]*>(.*?)<\/base>/is',
        '/<on\w+\s*=\s*["\'][^"\']*["\']/i',
        '/javascript:/i',
        '/data:/i',
        '/vbscript:/i',
        '/expression\s*\(/i',
        '/eval\s*\(/i',
        '/document\./i',
        '/window\./i',
        '/alert\s*\(/i',
        '/confirm\s*\(/i',
        '/prompt\s*\(/i',
    ];

    protected $sqlPatterns = [
        '/\b(SELECT|INSERT|UPDATE|DELETE|DROP|UNION|ALTER|CREATE|TRUNCATE)\b/i',
        '/\b(AND|OR|NOT|LIKE|IN|BETWEEN|EXISTS)\b/i',
        '/\b(WHERE|FROM|JOIN|GROUP BY|ORDER BY|HAVING)\b/i',
        '/\b(COUNT|SUM|AVG|MAX|MIN)\b/i',
        '/\b(ASC|DESC|LIMIT|OFFSET)\b/i',
        '/\b(PRIMARY|FOREIGN|KEY|INDEX|UNIQUE)\b/i',
        '/\b(INNER|LEFT|RIGHT|FULL|CROSS)\b/i',
        '/\b(ON|AS|IS|NULL|NOT NULL)\b/i',
        '/\b(BEGIN|COMMIT|ROLLBACK|TRANSACTION)\b/i',
        '/\b(GRANT|REVOKE|DENY)\b/i',
    ];

    public function handle(Request $request, Closure $next): Response
    {
        // پاکسازی داده‌های GET
        $this->sanitizeGetData($request);

        // پاکسازی داده‌های POST
        $this->sanitizePostData($request);

        // پاکسازی داده‌های JSON
        $this->sanitizeJsonData($request);

        // پاکسازی هدرها
        $this->sanitizeHeaders($request);

        return $next($request);
    }

    protected function sanitizeGetData(Request $request): void
    {
        $getData = $request->query();
        foreach ($getData as $key => $value) {
            if (is_string($value)) {
                $request->query->set($key, $this->sanitizeValue($value));
            }
        }
    }

    protected function sanitizePostData(Request $request): void
    {
        $postData = $request->request;
        foreach ($postData as $key => $value) {
            if (is_string($value)) {
                $request->request->set($key, $this->sanitizeValue($value));
            }
        }
    }

    protected function sanitizeJsonData(Request $request): void
    {
        if ($request->isJson()) {
            $jsonData = $request->json()->all();
            $sanitizedData = $this->sanitizeArray($jsonData);
            $request->json->replace($sanitizedData);
        }
    }

    protected function sanitizeHeaders(Request $request): void
    {
        $headers = $request->headers;
        foreach ($headers as $key => $value) {
            if (is_string($value)) {
                $headers->set($key, $this->sanitizeValue($value));
            }
        }
    }

    protected function sanitizeValue(string $value): string
    {
        // حذف کاراکترهای کنترل
        $value = preg_replace('/[\x00-\x1F\x7F]/u', '', $value);

        // حذف اسکریپت‌های مخرب
        foreach ($this->xssPatterns as $pattern) {
            $value = preg_replace($pattern, '', $value);
        }

        // حذف دستورات SQL
        foreach ($this->sqlPatterns as $pattern) {
            $value = preg_replace($pattern, '', $value);
        }

        // تبدیل کاراکترهای HTML
        $value = htmlspecialchars($value, ENT_QUOTES | ENT_HTML5, 'UTF-8');

        // حذف فاصله‌های اضافی
        $value = trim($value);

        return $value;
    }

    protected function sanitizeArray(array $data): array
    {
        $result = [];
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $result[$key] = $this->sanitizeArray($value);
            } elseif (is_string($value)) {
                $result[$key] = $this->sanitizeValue($value);
            } else {
                $result[$key] = $value;
            }
        }

        return $result;
    }
}
