<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class ClickjackingProtection
{
    /**
     * دامنه‌های مجاز برای نمایش در iframe
     */
    protected $allowedDomains = [
        'localhost',
        '127.0.0.1',
        // دامنه‌های مجاز خود را اینجا اضافه کنید
    ];

    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);

        // Enhanced X-Frame-Options protection based on route
        if ($request->is('admin/*')) {
            $response->headers->set('X-Frame-Options', 'DENY');
        } else {
            $response->headers->set('X-Frame-Options', 'SAMEORIGIN');
        }

        // Enhanced Content Security Policy for frame protection
        $frameAncestors = $request->is('admin/*') ? "'none'" : "'self' " . $this->getFrameAncestors();
        $existingCSP = $response->headers->get('Content-Security-Policy', '');

        if ($existingCSP) {
            // Add frame-ancestors to existing CSP if not present
            if (!str_contains($existingCSP, 'frame-ancestors')) {
                $response->headers->set('Content-Security-Policy', $existingCSP . "; frame-ancestors " . $frameAncestors);
            }
        } else {
            $response->headers->set('Content-Security-Policy', "frame-ancestors " . $frameAncestors);
        }

        // Additional security headers
        $response->headers->set('X-Content-Type-Options', 'nosniff');
        $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');

        // Log potential clickjacking attempts
        $this->logSuspiciousActivity($request);

        return $response;
    }

    protected function getFrameAncestors(): string
    {
        $domains = array_map(function ($domain) {
            return "https://{$domain}";
        }, $this->allowedDomains);

        return implode(' ', $domains);
    }

    protected function logSuspiciousActivity(Request $request): void
    {
        $referer = $request->header('Referer');

        // Check for suspicious referer patterns
        if ($referer && !str_starts_with($referer, $request->getSchemeAndHttpHost())) {
            $suspiciousPatterns = [
                'iframe',
                'frame',
                'embed',
                'object',
                'clickjack'
            ];

            foreach ($suspiciousPatterns as $pattern) {
                if (str_contains(strtolower($referer), $pattern)) {
                    \Log::warning('Potential clickjacking attempt detected', [
                        'ip' => $request->ip(),
                        'referer' => $referer,
                        'user_agent' => $request->userAgent(),
                        'url' => $request->fullUrl(),
                        'pattern' => $pattern
                    ]);
                    break;
                }
            }
        }

        // Check for suspicious headers
        $suspiciousHeaders = [
            'X-Frame-Options',
            'X-Forwarded-For',
            'X-Real-IP'
        ];

        foreach ($suspiciousHeaders as $header) {
            if ($request->hasHeader($header)) {
                \Log::info('Suspicious header detected', [
                    'ip' => $request->ip(),
                    'header' => $header,
                    'value' => $request->header($header),
                    'url' => $request->fullUrl()
                ]);
            }
        }
    }
}
