<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class CheckOrigin
{
    public function handle(Request $request, Closure $next): Response
    {
        $origin = $request->header('Origin');
        $host = $request->getHost();

        if ($request->isMethod('GET')) {
            return $next($request);
        }

        if (empty($origin)) {
            abort(403, 'درخواست بدون Origin مجاز نیست.');
        }

        if (! $this->isValidOrigin($origin, $host)) {
            abort(403, 'درخواست از این Origin مجاز نیست.');
        }

        return $next($request);
    }

    protected function isValidOrigin(?string $origin, string $host): bool
    {
        if (empty($origin)) {
            return false;
        }

        $originHost = parse_url($origin, PHP_URL_HOST);

        if (empty($originHost)) {
            return false;
        }

        // Allow requests from the same host
        if ($originHost === $host) {
            return true;
        }

        // Allow requests from trusted domains
        $trustedDomains = config('app.trusted_domains', []);
        if (in_array($originHost, $trustedDomains)) {
            return true;
        }

        // Check if the origin is a subdomain of the main domain
        if (str_ends_with($originHost, '.'.$host)) {
            return true;
        }

        return false;
    }
}
