<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Models\Location;
use App\Models\Category;
use Carbon\Carbon;

class SitemapController extends Controller
{
    public function index()
    {
        $sitemap = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $sitemap .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        
        // Main sitemap
        $sitemap .= $this->addSitemapEntry(url('/sitemap/main.xml'), Carbon::now());
        
        // Locations sitemap
        $sitemap .= $this->addSitemapEntry(url('/sitemap/locations.xml'), Carbon::now());
        
        // Categories sitemap
        $sitemap .= $this->addSitemapEntry(url('/sitemap/categories.xml'), Carbon::now());
        
        $sitemap .= '</sitemapindex>';
        
        return response($sitemap, 200, [
            'Content-Type' => 'application/xml',
            'Cache-Control' => 'public, max-age=3600'
        ]);
    }
    
    public function main()
    {
        $sitemap = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:xhtml="http://www.w3.org/1999/xhtml">' . "\n";
        
        // Homepage
        $sitemap .= $this->addUrlEntry(url('/'), '1.0', 'daily', Carbon::now());
        
        // Static pages
        $staticPages = [
            '/about' => ['priority' => '0.8', 'changefreq' => 'monthly'],
            '/contact' => ['priority' => '0.7', 'changefreq' => 'monthly'],
            '/map' => ['priority' => '0.9', 'changefreq' => 'weekly'],
            '/popular' => ['priority' => '0.8', 'changefreq' => 'daily'],
            '/categories' => ['priority' => '0.8', 'changefreq' => 'weekly'],
        ];
        
        foreach ($staticPages as $page => $config) {
            $sitemap .= $this->addUrlEntry(
                url($page), 
                $config['priority'], 
                $config['changefreq'], 
                Carbon::now()
            );
            
            // Add alternate language versions
            $sitemap .= $this->addUrlEntry(
                url($page . '?lang=en'), 
                $config['priority'], 
                $config['changefreq'], 
                Carbon::now(),
                [
                    'fa' => url($page),
                    'en' => url($page . '?lang=en')
                ]
            );
        }
        
        $sitemap .= '</urlset>';
        
        return response($sitemap, 200, [
            'Content-Type' => 'application/xml',
            'Cache-Control' => 'public, max-age=3600'
        ]);
    }
    
    public function locations()
    {
        $sitemap = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . "\n";
        
        $locations = Location::where('status', 'active')
            ->select('id', 'slug', 'updated_at', 'latitude', 'longitude')
            ->orderBy('updated_at', 'desc')
            ->get();
        
        foreach ($locations as $location) {
            $lastmod = $location->updated_at ?: Carbon::now();
            $priority = '0.7';
            
            // Higher priority for locations with coordinates
            if ($location->latitude && $location->longitude) {
                $priority = '0.8';
            }
            
            $url = url('/location/' . ($location->slug ?: $location->id));
            $sitemap .= $this->addUrlEntry($url, $priority, 'weekly', $lastmod);
        }
        
        $sitemap .= '</urlset>';
        
        return response($sitemap, 200, [
            'Content-Type' => 'application/xml',
            'Cache-Control' => 'public, max-age=1800'
        ]);
    }
    
    public function categories()
    {
        $sitemap = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        
        $categories = Category::where('status', 'active')
            ->select('id', 'slug', 'updated_at', 'parent_id')
            ->orderBy('updated_at', 'desc')
            ->get();
        
        foreach ($categories as $category) {
            $lastmod = $category->updated_at ?: Carbon::now();
            $priority = $category->parent_id ? '0.6' : '0.7'; // Parent categories have higher priority
            
            $url = url('/category/' . ($category->slug ?: $category->id));
            $sitemap .= $this->addUrlEntry($url, $priority, 'weekly', $lastmod);
        }
        
        $sitemap .= '</urlset>';
        
        return response($sitemap, 200, [
            'Content-Type' => 'application/xml',
            'Cache-Control' => 'public, max-age=3600'
        ]);
    }
    
    private function addSitemapEntry($loc, $lastmod)
    {
        return sprintf(
            "  <sitemap>\n    <loc>%s</loc>\n    <lastmod>%s</lastmod>\n  </sitemap>\n",
            htmlspecialchars($loc),
            $lastmod->toISOString()
        );
    }
    
    private function addUrlEntry($loc, $priority, $changefreq, $lastmod, $alternates = [])
    {
        $entry = "  <url>\n";
        $entry .= sprintf("    <loc>%s</loc>\n", htmlspecialchars($loc));
        $entry .= sprintf("    <lastmod>%s</lastmod>\n", $lastmod->toISOString());
        $entry .= sprintf("    <changefreq>%s</changefreq>\n", $changefreq);
        $entry .= sprintf("    <priority>%s</priority>\n", $priority);
        
        // Add alternate language links
        foreach ($alternates as $lang => $url) {
            $entry .= sprintf(
                '    <xhtml:link rel="alternate" hreflang="%s" href="%s" />' . "\n",
                $lang,
                htmlspecialchars($url)
            );
        }
        
        $entry .= "  </url>\n";
        
        return $entry;
    }
}
