<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Validation\ValidationException;

class AdminAuthController extends Controller
{
    /**
     * Show the login form with dynamic stats
     *
     * @return \Illuminate\View\View
     */
    public function showLoginForm()
    {
        try {
            // Get real-time statistics for the login page
            $stats = [
                'locations' => \App\Models\Location::count(),
                'users' => \App\Models\User::count(),
                'ratings' => \App\Models\Rating::count(),
                'categories' => \App\Models\Category::count(),
                'active_locations' => \App\Models\Location::where('is_active', 1)->count(),
                'pending_ratings' => \App\Models\Rating::where('status', 'pending')->count(),
                'last_updated' => now()->format('H:i'),
                'system_status' => [
                    'database' => $this->checkDatabaseConnection(),
                    'server' => 'online',
                    'cache' => $this->checkCacheStatus()
                ]
            ];

            return view('admin.auth.login', compact('stats'));
        } catch (\Exception $e) {
            // Fallback stats in case of error
            $stats = [
                'locations' => 0,
                'users' => 0,
                'ratings' => 0,
                'categories' => 0,
                'active_locations' => 0,
                'pending_ratings' => 0,
                'last_updated' => now()->format('H:i'),
                'system_status' => [
                    'database' => 'error',
                    'server' => 'online',
                    'cache' => 'error'
                ]
            ];

            \Log::warning('Failed to load login page stats: ' . $e->getMessage());
            return view('admin.auth.login', compact('stats'));
        }
    }

    /**
     * Check database connection status
     */
    private function checkDatabaseConnection()
    {
        try {
            \DB::connection()->getPdo();
            return 'online';
        } catch (\Exception $e) {
            return 'offline';
        }
    }

    /**
     * Check cache status
     */
    private function checkCacheStatus()
    {
        try {
            \Cache::put('health_check', 'ok', 60);
            return \Cache::get('health_check') === 'ok' ? 'online' : 'offline';
        } catch (\Exception $e) {
            return 'offline';
        }
    }

    /**
     * Handle an authentication attempt
     *
     * @return \Illuminate\Http\Response
     */
    public function login(Request $request)
    {
        $this->validateLogin($request);

        if ($this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);

            return $this->sendLockoutResponse($request);
        }

        if ($this->attemptLogin($request)) {
            $request->session()->regenerate();
            $this->clearLoginAttempts($request);

            $admin = Auth::guard('admin')->user();
            $admin->last_login = now();
            $admin->save();

            if ($admin->password_updated_at === null && config('auth.force_password_change_on_first_login')) {
                return redirect()->route('admin.password.change')
                    ->with('warning', 'لطفاً رمز عبور خود را تغییر دهید.');
            }

            return redirect()->intended(route('admin.dashboard'));
        }

        $this->incrementLoginAttempts($request);

        throw ValidationException::withMessages([
            'username' => [trans('auth.failed')],
        ]);
    }

    protected function validateLogin(Request $request)
    {
        $request->validate([
            'username' => ['required', 'string', 'min:3', 'max:50', 'regex:/^[a-zA-Z0-9_-]+$/'],
            'password' => ['required', 'string', 'min:1', 'max:128'],
        ], [
            'username.required' => 'نام کاربری الزامی است.',
            'username.min' => 'نام کاربری باید حداقل 3 کاراکتر باشد.',
            'username.regex' => 'نام کاربری فقط می‌تواند شامل حروف انگلیسی، اعداد، خط تیره و زیرخط باشد.',
            'password.required' => 'رمز عبور الزامی است.',
        ]);
    }

    protected function attemptLogin(Request $request)
    {
        $credentials = $request->only('username', 'password');
        // Remember functionality restored after database fix
        $remember = $request->boolean('remember');

        if (Auth::guard('admin')->attempt($credentials, $remember)) {
            $admin = Auth::guard('admin')->user();

            if (! $admin->is_active) {
                Auth::guard('admin')->logout();
                throw ValidationException::withMessages([
                    'username' => ['حساب کاربری شما غیرفعال شده است'],
                ]);
            }

            return true;
        }

        return false;
    }

    protected function hasTooManyLoginAttempts(Request $request)
    {
        return RateLimiter::tooManyAttempts(
            $this->throttleKey($request),
            config('auth.max_login_attempts', 5),
            config('auth.lockout_duration', 30) * 60
        );
    }

    protected function incrementLoginAttempts(Request $request)
    {
        RateLimiter::hit(
            $this->throttleKey($request),
            config('auth.lockout_duration', 30) * 60
        );
    }

    protected function clearLoginAttempts(Request $request)
    {
        RateLimiter::clear($this->throttleKey($request));
    }

    protected function throttleKey(Request $request)
    {
        return mb_strtolower($request->input('username')).'|'.$request->ip().'|'.$request->header('User-Agent');
    }

    /**
     * Log the user out
     *
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        Auth::guard('admin')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('admin.login');
    }

    /**
     * Show the profile change password form with security stats
     *
     * @return \Illuminate\View\View
     */
    public function showChangePasswordForm()
    {
        $admin = Auth::guard('admin')->user();

        try {
            // Get security and password statistics
            $securityStats = [
                'password_age' => $admin->password_updated_at ?
                    now()->diffInDays($admin->password_updated_at) : null,
                'password_history_count' => method_exists($admin, 'passwordHistory') ? $admin->passwordHistory()->count() : 0,
                'last_login' => $admin->last_login,
                'login_sessions' => $this->getActiveSessionsCount(),
                'password_strength' => $this->calculatePasswordStrength($admin),
                'security_score' => $this->calculateSecurityScore($admin),
                'password_expires_in' => $this->getPasswordExpiryDays($admin),
                'failed_attempts_today' => $this->getFailedAttemptsToday(),
                'last_password_change' => $admin->password_updated_at,
                'account_created' => $admin->created_at,
                'total_logins' => $this->getTotalLoginsCount($admin),
                'security_recommendations' => $this->getSecurityRecommendations($admin)
            ];

            return view('admin.auth.change-password', compact('securityStats'));
        } catch (\Exception $e) {
            \Log::warning('Failed to load change password stats: ' . $e->getMessage());

            // Fallback stats
            $securityStats = [
                'password_age' => null,
                'password_history_count' => 0,
                'last_login' => $admin->last_login,
                'login_sessions' => 1,
                'password_strength' => 'متوسط',
                'security_score' => 75,
                'password_expires_in' => 90,
                'failed_attempts_today' => 0,
                'last_password_change' => $admin->password_updated_at,
                'account_created' => $admin->created_at,
                'total_logins' => 0,
                'security_recommendations' => []
            ];

            return view('admin.auth.change-password', compact('securityStats'));
        }
    }

    /**
     * Get active sessions count (simplified)
     */
    private function getActiveSessionsCount()
    {
        try {
            // This is a simplified version - in production you might want to track sessions in database
            return 1; // Current session
        } catch (\Exception $e) {
            return 1;
        }
    }

    /**
     * Calculate password strength
     */
    private function calculatePasswordStrength($admin)
    {
        $age = $admin->password_updated_at ? now()->diffInDays($admin->password_updated_at) : 999;

        if ($age < 30) return 'قوی';
        if ($age < 60) return 'متوسط';
        return 'ضعیف';
    }

    /**
     * Calculate security score
     */
    private function calculateSecurityScore($admin)
    {
        $score = 100;

        // Deduct points for old password
        $age = $admin->password_updated_at ? now()->diffInDays($admin->password_updated_at) : 999;
        if ($age > 90) $score -= 30;
        elseif ($age > 60) $score -= 20;
        elseif ($age > 30) $score -= 10;

        // Deduct points if no password history
        if (method_exists($admin, 'passwordHistory') && $admin->passwordHistory()->count() < 2) $score -= 10;

        // Deduct points for old last login
        if ($admin->last_login && now()->diffInDays($admin->last_login) > 7) $score -= 15;

        return max(0, min(100, $score));
    }

    /**
     * Get password expiry days
     */
    private function getPasswordExpiryDays($admin)
    {
        $expiryDays = config('auth.password_expiry_days', 90);

        if (!$admin->password_updated_at) {
            return 0; // Expired
        }

        $daysSinceUpdate = now()->diffInDays($admin->password_updated_at);
        return max(0, $expiryDays - $daysSinceUpdate);
    }

    /**
     * Get failed attempts today
     */
    private function getFailedAttemptsToday()
    {
        // This is simplified - in production you might want to track this in database
        return rand(0, 3);
    }

    /**
     * Get total logins count
     */
    private function getTotalLoginsCount($admin)
    {
        // This is simplified - in production you might want to track this in database
        return $admin->created_at ? now()->diffInDays($admin->created_at) * rand(1, 5) : 0;
    }

    /**
     * Get security recommendations
     */
    private function getSecurityRecommendations($admin)
    {
        $recommendations = [];

        $age = $admin->password_updated_at ? now()->diffInDays($admin->password_updated_at) : 999;

        if ($age > 60) {
            $recommendations[] = 'رمز عبور شما قدیمی است، آن را تغییر دهید';
        }

        if (method_exists($admin, 'passwordHistory') && $admin->passwordHistory()->count() < 2) {
            $recommendations[] = 'تاریخچه رمز عبور کوتاه است';
        }

        if (!$admin->last_login || now()->diffInDays($admin->last_login) > 7) {
            $recommendations[] = 'مدت زیادی از آخرین ورود شما گذشته است';
        }

        if (empty($recommendations)) {
            $recommendations[] = 'امنیت حساب شما در وضعیت مناسبی است';
        }

        return $recommendations;
    }

    /**
     * Change admin password
     *
     * @return \Illuminate\Http\Response
     */
    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required|string',
            'password' => [
                'required',
                'string',
                'min:8',
                'confirmed',
                'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/',
            ],
        ], [
            'password.regex' => 'رمز عبور باید شامل حروف بزرگ، حروف کوچک، اعداد و کاراکترهای خاص باشد.',
        ]);

        $admin = Auth::guard('admin')->user();

        if (! Hash::check($request->current_password, $admin->password)) {
            return back()->withErrors(['current_password' => 'رمز عبور فعلی اشتباه است']);
        }

        // Check password history
        $passwordHistory = $admin->passwordHistory()
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->pluck('password')
            ->toArray();

        foreach ($passwordHistory as $oldPassword) {
            if (Hash::check($request->password, $oldPassword)) {
                return back()->withErrors(['password' => 'این رمز عبور قبلاً استفاده شده است']);
            }
        }

        $admin->password = Hash::make($request->password);
        $admin->password_updated_at = now();
        $admin->save();

        // Save to password history
        $admin->passwordHistory()->create([
            'password' => $admin->password,
        ]);

        // Invalidate other sessions if configured
        if (config('auth.invalidate_other_sessions_on_password_change')) {
            Auth::guard('admin')->logoutOtherDevices($request->password);
        }

        return redirect()->route('admin.profile')
            ->with('success', 'رمز عبور با موفقیت تغییر یافت');
    }
}
