<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\City;
use App\Models\Country;
use App\Models\County;
use App\Models\District;
use App\Models\Location;
use App\Models\Province;
use App\Models\Village;
use App\Models\Zone;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ApiController extends Controller
{
    /**
     * Get locations data for map filtering
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLocations(Request $request)
    {
        try {
            // Simplified query for better performance and reliability
            $query = Location::where('is_active', 1);

            // Only get locations with valid coordinates
            $query->whereNotNull('lat')
                  ->whereNotNull('lng')
                  ->where('lat', '<>', 0)
                  ->where('lng', '<>', 0);

            // Basic filters
            if ($request->filled('category_id')) {
                $query->where('category_id', $request->category_id);
            }

            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('title', 'LIKE', "%{$search}%")
                      ->orWhere('description', 'LIKE', "%{$search}%")
                      ->orWhere('address', 'LIKE', "%{$search}%");
                });
            }

            // Simple sorting
            $query->orderBy('created_at', 'desc');

            // Pagination - allow unlimited for map display
            if ($request->has('limit')) {
                $limit = min($request->get('limit', 1000), 2000); // Max 2000 locations for performance
                $offset = $request->get('offset', 0);
                $locations = $query->offset($offset)->limit($limit)->get();
            } else {
                // No limit - get all locations for map
                $locations = $query->get();
            }

            // Simplified transform for map display
            $transformedLocations = $locations->map(function($location) {
                return [
                    'id' => $location->id,
                    'title_id' => $location->title_id,
                    'title' => $location->title,
                    'description' => $location->description,
                    'lat' => (float) $location->lat,
                    'lng' => (float) $location->lng,
                    'address' => $location->address,
                    'phone' => $location->phone,
                    'category_id' => $location->category_id,
                    'category_fa' => $location->category_fa,
                    'category_color' => '#2563eb',
                    'category_emoji' => '📍',
                    'package_type' => $location->package_type ?? 'regular',
                    'average_rating' => $location->star_rating ? (float) $location->star_rating : 0,
                    'total_ratings' => 0,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedLocations,
                'total' => $transformedLocations->count(),
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getLocations API:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات مکان‌ها',
                'error' => config('app.debug') ? $e->getMessage() : 'خطای داخلی سرور'
            ], 500);
        }
    }

    /**
     * Calculate distance between two points using Haversine formula
     */
    private function calculateDistance($lat1, $lng1, $lat2, $lng2)
    {
        $earthRadius = 6371; // Earth's radius in kilometers

        $dLat = deg2rad($lat2 - $lat1);
        $dLng = deg2rad($lng2 - $lng1);

        $a = sin($dLat/2) * sin($dLat/2) +
             cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
             sin($dLng/2) * sin($dLng/2);

        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        $distance = $earthRadius * $c;

        return $distance;
    }

    /**
     * Simple test endpoint
     */
    public function test()
    {
        try {
            $count = Location::count();
            $activeCount = Location::where('is_active', 1)->count();
            $withCoordinates = Location::where('is_active', 1)
                ->whereNotNull('lat')
                ->whereNotNull('lng')
                ->where('lat', '<>', 0)
                ->where('lng', '<>', 0)
                ->count();

            $sampleLocation = Location::where('is_active', 1)->first();

            return response()->json([
                'success' => true,
                'total_locations' => $count,
                'active_locations' => $activeCount,
                'locations_with_coordinates' => $withCoordinates,
                'sample_location' => $sampleLocation ? [
                    'id' => $sampleLocation->id,
                    'title' => $sampleLocation->title,
                    'lat' => $sampleLocation->lat,
                    'lng' => $sampleLocation->lng,
                    'category_fa' => $sampleLocation->category_fa,
                    'is_active' => $sampleLocation->is_active
                ] : null
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], 500);
        }
    }

    /**
     * Simple locations endpoint for map
     */
    public function getSimpleLocations()
    {
        try {
            $locations = Location::where('locations.is_active', 1)
                ->whereNotNull('locations.lat')
                ->whereNotNull('locations.lng')
                ->where('locations.lat', '<>', 0)
                ->where('locations.lng', '<>', 0)
                ->leftJoin('categories', 'locations.category_id', '=', 'categories.id')
                ->select([
                    'locations.id',
                    'locations.title',
                    'locations.lat',
                    'locations.lng',
                    'locations.address',
                    'locations.phone',
                    'locations.category_fa',
                    'locations.category_id',
                    'locations.package_type',
                    'categories.emoji as category_emoji',
                    'categories.color as category_color',
                    'categories.font_icon as category_icon'
                ])
                ->get();

            // Transform data to include proper emoji and color
            $transformedLocations = $locations->map(function($location) {
                return [
                    'id' => $location->id,
                    'title' => $location->title,
                    'lat' => (float) $location->lat,
                    'lng' => (float) $location->lng,
                    'address' => $location->address,
                    'phone' => $location->phone,
                    'category_id' => $location->category_id,
                    'category_fa' => $location->category_fa,
                    'category_emoji' => $location->category_emoji ?: '📍',
                    'category_color' => $location->category_color ?: '#2563eb',
                    'category_icon' => $location->category_icon ?: 'fas fa-map-marker-alt',
                    'package_type' => $location->package_type ?: 'regular',
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedLocations,
                'total' => $transformedLocations->count()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }



    /**
     * Get categories for map filtering
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCategories(Request $request)
    {
        try {
            $query = \App\Models\Category::where('is_active', true)
                ->where('show_in_filters', true);

            // Get parent categories with their children and location counts
            if ($request->boolean('include_children', true)) {
                $categories = $query->whereNull('parent_id')
                    ->with(['children' => function($q) {
                        $q->where('is_active', true)
                          ->where('show_in_filters', true)
                          ->withCount(['locations' => function($query) {
                              $query->where('is_active', 1)
                                    ->whereNotNull('lat')
                                    ->whereNotNull('lng');
                          }])
                          ->orderBy('sort_order');
                    }])
                    ->withCount(['locations' => function($query) {
                        $query->where('is_active', 1)
                              ->whereNotNull('lat')
                              ->whereNotNull('lng');
                    }])
                    ->orderBy('sort_order')
                    ->get();
            } else {
                $categories = $query->withCount(['locations' => function($query) {
                    $query->where('is_active', 1)
                          ->whereNotNull('lat')
                          ->whereNotNull('lng');
                }])->orderBy('sort_order')->get();
            }

            // Transform data for frontend
            $transformedCategories = $categories->map(function($category) {
                $data = [
                    'id' => $category->id,
                    'category_key' => $category->category_key,
                    'category_fa' => $category->category_fa,
                    'category_en' => $category->category_en,
                    'category_ar' => $category->category_ar,
                    'description' => $category->description,
                    'emoji' => $category->emoji,
                    'font_icon' => $category->font_icon ?? 'fas fa-map-marker-alt',
                    'color' => $category->color ?? '#2563eb',
                    'sort_order' => $category->sort_order,
                    'parent_id' => $category->parent_id,
                    'locations_count' => $category->locations_count ?? 0,
                    'is_featured' => $category->is_featured ?? false,
                    'meta_title' => $category->meta_title,
                    'meta_description' => $category->meta_description,
                ];

                // Add children if they exist
                if ($category->children && $category->children->count() > 0) {
                    $data['children'] = $category->children->map(function($child) {
                        return [
                            'id' => $child->id,
                            'category_key' => $child->category_key,
                            'category_fa' => $child->category_fa,
                            'category_en' => $child->category_en,
                            'category_ar' => $child->category_ar,
                            'description' => $child->description,
                            'emoji' => $child->emoji,
                            'font_icon' => $child->font_icon ?? 'fas fa-map-marker-alt',
                            'color' => $child->color ?? '#2563eb',
                            'sort_order' => $child->sort_order,
                            'parent_id' => $child->parent_id,
                            'locations_count' => $child->locations_count ?? 0,
                            'is_featured' => $child->is_featured ?? false,
                        ];
                    });

                    // Calculate total locations for parent including children
                    $data['total_locations_count'] = $data['locations_count'] +
                        $data['children']->sum('locations_count');
                }

                return $data;
            });

            return response()->json([
                'success' => true,
                'data' => $transformedCategories,
                'total' => $transformedCategories->count(),
                'total_locations' => $transformedCategories->sum('total_locations_count')
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getCategories API:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت دسته‌بندی‌ها',
                'error' => config('app.debug') ? $e->getMessage() : 'خطای داخلی سرور'
            ], 500);
        }
    }

    /**
     * Get provinces for geographic filtering
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProvinces(Request $request)
    {
        try {
            $query = \App\Models\Province::where('is_active', true);

            if ($request->filled('country_id')) {
                $query->where('country_id', $request->country_id);
            }

            // Include location counts
            $provinces = $query->withCount(['locations' => function($query) {
                $query->where('is_active', 1)
                      ->whereNotNull('lat')
                      ->whereNotNull('lng');
            }])
            ->orderBy('name')
            ->get();

            $transformedProvinces = $provinces->map(function($province) {
                return [
                    'id' => $province->id,
                    'name' => $province->name,
                    'name_en' => $province->name_en ?? null,
                    'name_ar' => $province->name_ar ?? null,
                    'code' => $province->code,
                    'country_id' => $province->country_id,
                    'locations_count' => $province->locations_count ?? 0,
                    'is_capital' => $province->is_capital ?? false,
                    'latitude' => $province->latitude,
                    'longitude' => $province->longitude,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedProvinces,
                'total' => $transformedProvinces->count(),
                'total_locations' => $transformedProvinces->sum('locations_count')
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getProvinces API:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت استان‌ها',
                'error' => config('app.debug') ? $e->getMessage() : 'خطای داخلی سرور'
            ], 500);
        }
    }

    /**
     * Get cities by province for geographic filtering
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCitiesByProvince($provinceId)
    {
        try {
            $cities = \App\Models\City::where('is_active', true)
                ->whereHas('district.county.province', function($query) use ($provinceId) {
                    $query->where('id', $provinceId);
                })
                ->withCount(['locations' => function($query) {
                    $query->where('is_active', 1)
                          ->whereNotNull('lat')
                          ->whereNotNull('lng');
                }])
                ->orderBy('name')
                ->get();

            $transformedCities = $cities->map(function($city) {
                return [
                    'id' => $city->id,
                    'name' => $city->name,
                    'name_en' => $city->name_en ?? null,
                    'name_ar' => $city->name_ar ?? null,
                    'district_id' => $city->district_id,
                    'county_id' => $city->county_id,
                    'is_capital' => $city->is_capital ?? false,
                    'locations_count' => $city->locations_count ?? 0,
                    'latitude' => $city->latitude,
                    'longitude' => $city->longitude,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedCities,
                'total' => $transformedCities->count(),
                'total_locations' => $transformedCities->sum('locations_count')
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getCitiesByProvince API:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت شهرها',
                'error' => config('app.debug') ? $e->getMessage() : 'خطای داخلی سرور'
            ], 500);
        }
    }

    /**
     * Get counties by province
     */
    public function getCountiesByProvince($provinceId)
    {
        try {
            $counties = \App\Models\County::where('is_active', true)
                ->where('province_id', $provinceId)
                ->withCount(['locations' => function($query) {
                    $query->where('is_active', 1)
                          ->whereNotNull('lat')
                          ->whereNotNull('lng');
                }])
                ->orderBy('name')
                ->get();

            $transformedCounties = $counties->map(function($county) {
                return [
                    'id' => $county->id,
                    'name' => $county->name,
                    'name_fa' => $county->name, // فیلد name_fa وجود ندارد، از name استفاده می‌کنیم
                    'name_en' => $county->name_en ?? null,
                    'name_ar' => $county->name_ar ?? null,
                    'province_id' => $county->province_id,
                    'locations_count' => $county->locations_count ?? 0,
                    'latitude' => $county->latitude,
                    'longitude' => $county->longitude,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedCounties,
                'total' => $transformedCounties->count(),
                'total_locations' => $transformedCounties->sum('locations_count')
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getCountiesByProvince API:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت شهرستان‌ها',
                'error' => config('app.debug') ? $e->getMessage() : 'خطای داخلی سرور'
            ], 500);
        }
    }

    /**
     * Get districts by county
     */
    public function getDistrictsByCounty($countyId)
    {
        try {
            $districts = \App\Models\District::where('is_active', true)
                ->where('county_id', $countyId)
                ->withCount(['locations' => function($query) {
                    $query->where('is_active', 1)
                          ->whereNotNull('lat')
                          ->whereNotNull('lng');
                }])
                ->orderBy('name')
                ->get();

            $transformedDistricts = $districts->map(function($district) {
                return [
                    'id' => $district->id,
                    'name' => $district->name,
                    'name_fa' => $district->name, // فیلد name_fa وجود ندارد، از name استفاده می‌کنیم
                    'name_en' => $district->name_en ?? null,
                    'name_ar' => $district->name_ar ?? null,
                    'county_id' => $district->county_id,
                    'locations_count' => $district->locations_count ?? 0,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedDistricts,
                'total' => $transformedDistricts->count(),
                'total_locations' => $transformedDistricts->sum('locations_count')
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getDistrictsByCounty API:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت بخش‌ها',
                'error' => config('app.debug') ? $e->getMessage() : 'خطای داخلی سرور'
            ], 500);
        }
    }

    /**
     * Get cities by county
     */
    public function getCitiesByCounty($countyId)
    {
        try {
            $cities = \App\Models\City::where('is_active', true)
                ->where('county_id', $countyId)
                ->withCount(['locations' => function($query) {
                    $query->where('is_active', 1)
                          ->whereNotNull('lat')
                          ->whereNotNull('lng');
                }])
                ->orderBy('name')
                ->get();

            $transformedCities = $cities->map(function($city) {
                return [
                    'id' => $city->id,
                    'name' => $city->name,
                    'name_fa' => $city->name, // فیلد name_fa وجود ندارد، از name استفاده می‌کنیم
                    'name_en' => $city->name_en ?? null,
                    'name_ar' => $city->name_ar ?? null,
                    'county_id' => $city->county_id,
                    'is_capital' => $city->is_capital ?? false,
                    'locations_count' => $city->locations_count ?? 0,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedCities,
                'total' => $transformedCities->count(),
                'total_locations' => $transformedCities->sum('locations_count')
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getCitiesByCounty API:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت شهرها',
                'error' => config('app.debug') ? $e->getMessage() : 'خطای داخلی سرور'
            ], 500);
        }
    }

    /**
     * Get villages by district
     */
    public function getVillagesByDistrict($districtId)
    {
        try {
            $villages = \App\Models\Village::where('is_active', true)
                ->where('district_id', $districtId)
                ->withCount(['locations' => function($query) {
                    $query->where('is_active', 1)
                          ->whereNotNull('lat')
                          ->whereNotNull('lng');
                }])
                ->orderBy('name')
                ->get();

            $transformedVillages = $villages->map(function($village) {
                return [
                    'id' => $village->id,
                    'name' => $village->name,
                    'name_fa' => $village->name_fa ?? $village->name,
                    'name_en' => $village->name_en ?? null,
                    'name_ar' => $village->name_ar ?? null,
                    'district_id' => $village->district_id,
                    'locations_count' => $village->locations_count ?? 0,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedVillages,
                'total' => $transformedVillages->count(),
                'total_locations' => $transformedVillages->sum('locations_count')
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getVillagesByDistrict API:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت روستاها',
                'error' => config('app.debug') ? $e->getMessage() : 'خطای داخلی سرور'
            ], 500);
        }
    }

    /**
     * Get zones by city
     */
    public function getZonesByCity($cityId)
    {
        try {
            $zones = \App\Models\Zone::where('is_active', true)
                ->where('city_id', $cityId)
                ->withCount(['locations' => function($query) {
                    $query->where('is_active', 1)
                          ->whereNotNull('lat')
                          ->whereNotNull('lng');
                }])
                ->orderBy('name')
                ->get();

            $transformedZones = $zones->map(function($zone) {
                return [
                    'id' => $zone->id,
                    'name' => $zone->name,
                    'name_fa' => $zone->name_fa ?? $zone->name,
                    'name_en' => $zone->name_en ?? null,
                    'name_ar' => $zone->name_ar ?? null,
                    'city_id' => $zone->city_id,
                    'zone_type' => $zone->zone_type ?? null,
                    'locations_count' => $zone->locations_count ?? 0,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedZones,
                'total' => $transformedZones->count(),
                'total_locations' => $transformedZones->sum('locations_count')
            ]);

        } catch (\Exception $e) {
            Log::error('Error in getZonesByCity API:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت مناطق',
                'error' => config('app.debug') ? $e->getMessage() : 'خطای داخلی سرور'
            ], 500);
        }
    }
}
