<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Amenity;
use App\Models\AmenityType;
use Illuminate\Http\Request;

class AmenityController extends Controller
{
    /**
     * Display a listing of the amenity types.
     */
    public function index()
    {
        // Check if we're showing amenity types or amenities based on the route
        if (request()->route()->getName() === 'admin.amenity-types.index') {
            $amenityTypes = AmenityType::orderBy('display_order')->paginate(15);

            return view('admin.amenities.types.index', compact('amenityTypes'));
        } else {
            $amenities = Amenity::with('type')->orderBy('display_order')->paginate(15);

            return view('admin.amenities.index', compact('amenities'));
        }
    }

    /**
     * Show the form for creating a new amenity type.
     */
    public function create()
    {
        // Check if we're creating an amenity type or an amenity based on the route
        if (request()->route()->getName() === 'admin.amenity-types.create') {
            return view('admin.amenities.types.create');
        } else {
            $types = AmenityType::orderBy('display_order')->get();
            
            // Debug information
            \Log::info('Route name: ' . request()->route()->getName());
            \Log::info('View path: admin.amenities.create');
            \Log::info('Types count: ' . $types->count());
            \Log::info('View exists: ' . view()->exists('admin.amenities.create'));
            \Log::info('Layout exists: ' . view()->exists('admin.layouts.app'));
            \Log::info('View path: ' . resource_path('views/admin/amenities/create.blade.php'));
            \Log::info('Layout path: ' . resource_path('views/admin/layouts/app.blade.php'));
            
            try {
                return view('admin.amenities.create', compact('types'));
            } catch (\Exception $e) {
                \Log::error('View error: ' . $e->getMessage());
                \Log::error('Stack trace: ' . $e->getTraceAsString());
                throw $e;
            }
        }
    }

    /**
     * Store a newly created amenity type in storage.
     */
    public function store(Request $request)
    {
        // Check if we're storing an amenity type or an amenity based on the route
        if (request()->route()->getName() === 'admin.amenity-types.store') {
            $validated = $request->validate([
                'type_key' => 'required|string|max:50|unique:amenity_types',
                'type_name' => 'required|string|max:100',
                'type_section' => 'required|string',
                'display_order' => 'nullable|integer',
            ]);

            AmenityType::create($validated);

            return redirect()->route('admin.amenity-types.index')
                ->with('success', 'نوع امکان با موفقیت ایجاد شد');
        } else {
            $validated = $request->validate([
                'type_id' => 'required|exists:amenity_types,id',
                'title' => 'required|string|max:100',
                'icon' => 'nullable|string|max:100',
                'description' => 'nullable|string',
                'display_order' => 'nullable|integer',
                'is_active' => 'nullable|boolean',
            ]);

            Amenity::create($validated);

            return redirect()->route('admin.amenities.index')
                ->with('success', 'امکان با موفقیت ایجاد شد');
        }
    }

    /**
     * Display the specified amenity type.
     */
    public function show($id)
    {
        // Only for amenity types
        $amenityType = AmenityType::findOrFail($id);
        $amenities = Amenity::where('type_id', $id)->orderBy('display_order')->get();

        return view('admin.amenities.types.show', compact('amenityType', 'amenities'));
    }

    /**
     * Show the form for editing the specified amenity type.
     */
    public function edit($id)
    {
        // Check if we're editing an amenity type or an amenity based on the route
        if (request()->route()->getName() === 'admin.amenity-types.edit') {
            $amenityType = AmenityType::findOrFail($id);

            return view('admin.amenities.types.edit', compact('amenityType'));
        } else {
            $amenity = Amenity::findOrFail($id);
            $types = AmenityType::orderBy('display_order')->get();

            return view('admin.amenities.edit', compact('amenity', 'types'));
        }
    }

    /**
     * Update the specified amenity type in storage.
     */
    public function update(Request $request, $id)
    {
        // Check if we're updating an amenity type or an amenity based on the route
        if (request()->route()->getName() === 'admin.amenity-types.update') {
            $amenityType = AmenityType::findOrFail($id);

            $validated = $request->validate([
                'type_key' => 'required|string|max:50|unique:amenity_types,type_key,'.$id,
                'type_name' => 'required|string|max:100',
                'type_section' => 'required|string',
                'display_order' => 'nullable|integer',
            ]);

            $amenityType->update($validated);

            return redirect()->route('admin.amenity-types.index')
                ->with('success', 'نوع امکان با موفقیت بروزرسانی شد');
        } else {
            $amenity = Amenity::findOrFail($id);

            $validated = $request->validate([
                'type_id' => 'required|exists:amenity_types,id',
                'title' => 'required|string|max:100',
                'icon' => 'nullable|string|max:100',
                'description' => 'nullable|string',
                'display_order' => 'nullable|integer',
                'is_active' => 'nullable|boolean',
            ]);

            $amenity->update($validated);

            return redirect()->route('admin.amenities.index')
                ->with('success', 'امکان با موفقیت بروزرسانی شد');
        }
    }

    /**
     * Remove the specified amenity type from storage.
     */
    public function destroy($id)
    {
        // Check if we're deleting an amenity type or an amenity based on the route
        if (request()->route()->getName() === 'admin.amenity-types.destroy') {
            $amenityType = AmenityType::findOrFail($id);

            // Check if there are amenities using this type
            $count = Amenity::where('type_id', $id)->count();
            if ($count > 0) {
                return redirect()->route('admin.amenity-types.index')
                    ->with('error', 'این نوع امکان دارای امکانات وابسته است و نمی‌توان آن را حذف کرد');
            }

            $amenityType->delete();

            return redirect()->route('admin.amenity-types.index')
                ->with('success', 'نوع امکان با موفقیت حذف شد');
        } else {
            $amenity = Amenity::findOrFail($id);
            $amenity->delete();

            return redirect()->route('admin.amenities.index')
                ->with('success', 'امکان با موفقیت حذف شد');
        }
    }
}
