<?php

namespace App\Helpers;

class BreadcrumbHelper
{
    /**
     * Generate breadcrumb based on current route.
     */
    public static function generate()
    {
        $breadcrumb = [];
        $currentRoute = request()->route();
        
        if (!$currentRoute) {
            return $breadcrumb;
        }
        
        $routeName = $currentRoute->getName();
        $uri = request()->getRequestUri();
        $segments = array_filter(explode('/', trim($uri, '/')));
        
        // Remove language code from segments if exists
        if (!empty($segments) && in_array($segments[0], ['fa', 'en'])) {
            array_shift($segments);
        }
        
        // Define route mappings
        $routeMappings = [
            // Admin routes
            'admin.dashboard' => ['title' => 'داشبورد مدیریت', 'url' => '/admin'],
            'admin.menu-management.index' => ['title' => 'مدیریت منوها', 'url' => '/admin/menu-management'],
            'admin.header-settings.index' => ['title' => 'تنظیمات هدر', 'url' => '/admin/header-settings'],
            
            // User routes
            'user.dashboard' => ['title' => 'داشبورد کاربری', 'url' => '/user/dashboard'],
            'user.profile' => ['title' => 'پروفایل کاربری', 'url' => '/user/profile'],
            'user.locations.index' => ['title' => 'مکان‌های من', 'url' => '/user/locations'],
            
            // Public routes
            'categories.index' => ['title' => 'دسته‌بندی‌ها', 'url' => '/categories'],
            'map.index' => ['title' => 'نقشه', 'url' => '/map'],
            'popular.index' => ['title' => 'مکان‌های محبوب', 'url' => '/popular'],
            'about.index' => ['title' => 'درباره ما', 'url' => '/about'],
            'contact.index' => ['title' => 'تماس با ما', 'url' => '/contact'],
            'search.index' => ['title' => 'نتایج جستجو', 'url' => '/search'],
            
            // Location routes
            'locations.show' => ['title' => 'جزئیات مکان', 'url' => null],
            'locations.create' => ['title' => 'ثبت مکان جدید', 'url' => '/add-place'],
            'locations.edit' => ['title' => 'ویرایش مکان', 'url' => null],
        ];
        
        // Generate breadcrumb based on route name
        if ($routeName && isset($routeMappings[$routeName])) {
            $mapping = $routeMappings[$routeName];
            
            // Add parent breadcrumbs for nested routes
            if (str_contains($routeName, 'admin.')) {
                if ($routeName !== 'admin.dashboard') {
                    $breadcrumb[] = ['title' => 'داشبورد مدیریت', 'url' => '/admin'];
                }
            } elseif (str_contains($routeName, 'user.')) {
                if ($routeName !== 'user.dashboard') {
                    $breadcrumb[] = ['title' => 'داشبورد کاربری', 'url' => '/user/dashboard'];
                }
            }
            
            // Add current page
            $breadcrumb[] = [
                'title' => $mapping['title'],
                'url' => $mapping['url']
            ];
        } else {
            // Fallback: generate from URL segments
            $breadcrumb = self::generateFromSegments($segments);
        }
        
        return $breadcrumb;
    }
    
    /**
     * Generate breadcrumb from URL segments.
     */
    private static function generateFromSegments($segments)
    {
        $breadcrumb = [];
        $url = '';
        
        $segmentMappings = [
            'admin' => 'داشبورد مدیریت',
            'user' => 'داشبورد کاربری',
            'categories' => 'دسته‌بندی‌ها',
            'map' => 'نقشه',
            'popular' => 'مکان‌های محبوب',
            'about' => 'درباره ما',
            'contact' => 'تماس با ما',
            'search' => 'جستجو',
            'locations' => 'مکان‌ها',
            'menu-management' => 'مدیریت منوها',
            'header-settings' => 'تنظیمات هدر',
            'profile' => 'پروفایل',
            'dashboard' => 'داشبورد',
            'add-place' => 'ثبت مکان جدید',
            'create' => 'ایجاد',
            'edit' => 'ویرایش',
            'show' => 'نمایش',
        ];
        
        foreach ($segments as $segment) {
            $url .= '/' . $segment;
            
            // Skip numeric IDs
            if (is_numeric($segment)) {
                continue;
            }
            
            $title = $segmentMappings[$segment] ?? ucfirst(str_replace('-', ' ', $segment));
            
            $breadcrumb[] = [
                'title' => $title,
                'url' => $url
            ];
        }
        
        return $breadcrumb;
    }
    
    /**
     * Get page title from breadcrumb.
     */
    public static function getPageTitle($breadcrumb = null)
    {
        if (!$breadcrumb) {
            $breadcrumb = self::generate();
        }
        
        if (empty($breadcrumb)) {
            return 'صفحه فعلی';
        }
        
        return end($breadcrumb)['title'];
    }
}
