/**
 * Locale Date Manager
 * مدیریت تاریخ‌ها بر اساس زبان در نمودارها و تقویم‌ها
 */

// جلوگیری از تعریف مجدد کلاس
if (typeof window.LocaleDateManager === 'undefined') {
    window.LocaleDateManager = class LocaleDateManager {
    constructor() {
        this.locale = document.documentElement.lang || 'fa';
        this.persianMonths = [
            'فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور',
            'مهر', 'آبان', 'آذر', 'دی', 'بهمن', 'اسفند'
        ];
        this.persianDays = [
            'شنبه', 'یکشنبه', 'دوشنبه', 'سه‌شنبه', 'چهارشنبه', 'پنج‌شنبه', 'جمعه'
        ];
        this.persianDaysShort = ['ش', 'ی', 'د', 'س', 'چ', 'پ', 'ج'];
        
        this.init();
    }

    init() {
        this.setupChartDefaults();
        this.initializeDatePickers();
        this.updateExistingCharts();
    }

    /**
     * تنظیم پیش‌فرض‌های Chart.js بر اساس زبان
     */
    setupChartDefaults() {
        if (typeof Chart !== 'undefined') {
            Chart.defaults.font.family = this.locale === 'fa' ? 'IRANSans, sans-serif' : 'Arial, sans-serif';
            Chart.defaults.plugins.legend.rtl = this.locale === 'fa';
            
            // تنظیم فرمت تاریخ پیش‌فرض
            Chart.defaults.scales.time = {
                ...Chart.defaults.scales.time,
                displayFormats: this.getChartDateFormats()
            };
        }
    }

    /**
     * دریافت فرمت‌های تاریخ برای نمودارها
     */
    getChartDateFormats() {
        if (this.locale === 'fa') {
            return {
                millisecond: 'HH:mm:ss.SSS',
                second: 'HH:mm:ss',
                minute: 'HH:mm',
                hour: 'HH:mm',
                day: 'Y/M/D',
                week: 'Y/M/D',
                month: 'Y/M',
                quarter: 'Y/M',
                year: 'Y'
            };
        } else {
            return {
                millisecond: 'HH:mm:ss.SSS',
                second: 'HH:mm:ss',
                minute: 'HH:mm',
                hour: 'HH:mm',
                day: 'MMM DD',
                week: 'MMM DD',
                month: 'MMM YYYY',
                quarter: 'MMM YYYY',
                year: 'YYYY'
            };
        }
    }

    /**
     * فرمت کردن تاریخ برای نمودارها
     */
    formatDateForChart(date) {
        if (!date) return '';
        
        const dateObj = new Date(date);
        
        if (this.locale === 'fa') {
            return this.toJalali(dateObj);
        } else {
            return dateObj.toLocaleDateString(this.locale);
        }
    }

    /**
     * فرمت کردن ماه برای نمودارها
     */
    formatMonthForChart(monthNumber) {
        if (this.locale === 'fa') {
            return this.persianMonths[monthNumber - 1] || monthNumber;
        } else {
            const date = new Date(2000, monthNumber - 1, 1);
            return date.toLocaleDateString(this.locale, { month: 'long' });
        }
    }

    /**
     * تبدیل تاریخ میلادی به شمسی
     */
    toJalali(date) {
        if (typeof jDate !== 'undefined') {
            return jDate(date).format('Y/M/D');
        }
        
        // Fallback: استفاده از API سرور
        return this.convertDateViaServer(date, 'jalali');
    }

    /**
     * تبدیل تاریخ شمسی به میلادی
     */
    toGregorian(jalaliDate) {
        if (typeof jDate !== 'undefined') {
            return jDate.jalaliToGregorian(jalaliDate);
        }
        
        // Fallback: استفاده از API سرور
        return this.convertDateViaServer(jalaliDate, 'gregorian');
    }

    /**
     * تبدیل اعداد به فارسی
     */
    toPersianNumbers(str) {
        if (this.locale !== 'fa') return str;
        
        const persianDigits = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
        return str.toString().replace(/\d/g, (digit) => persianDigits[parseInt(digit)]);
    }

    /**
     * تبدیل اعداد فارسی به انگلیسی
     */
    toEnglishNumbers(str) {
        const persianDigits = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
        const englishDigits = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
        
        let result = str.toString();
        for (let i = 0; i < persianDigits.length; i++) {
            result = result.replace(new RegExp(persianDigits[i], 'g'), englishDigits[i]);
        }
        return result;
    }

    /**
     * راه‌اندازی تقویم‌ها
     */
    initializeDatePickers() {
        // تقویم‌های موجود را به‌روزرسانی کن
        this.updateDatePickers();
        
        // رصد تغییرات DOM برای تقویم‌های جدید
        this.observeNewDatePickers();
    }

    /**
     * به‌روزرسانی تقویم‌های موجود
     */
    updateDatePickers() {
        const dateInputs = document.querySelectorAll('input[type="date"], .datepicker, .admin-datepicker');
        
        dateInputs.forEach(input => {
            this.setupDatePicker(input);
        });
    }

    /**
     * تنظیم تقویم بر اساس زبان
     */
    setupDatePicker(input) {
        if (input.dataset.localeSetup) return; // جلوگیری از تنظیم مجدد
        
        if (this.locale === 'fa') {
            this.setupPersianDatePicker(input);
        } else {
            this.setupGregorianDatePicker(input);
        }
        
        input.dataset.localeSetup = 'true';
    }

    /**
     * تنظیم تقویم فارسی
     */
    setupPersianDatePicker(input) {
        // اگر کتابخانه تقویم فارسی موجود است
        if (typeof $.fn.persianDatepicker !== 'undefined') {
            $(input).persianDatepicker({
                format: 'YYYY/MM/DD',
                initialValue: false,
                autoClose: true,
                calendar: {
                    persian: {
                        locale: 'fa',
                        showHint: true
                    }
                }
            });
        } else if (typeof Pikaday !== 'undefined') {
            new Pikaday({
                field: input,
                format: 'YYYY/MM/DD',
                i18n: {
                    previousMonth: 'ماه قبل',
                    nextMonth: 'ماه بعد',
                    months: this.persianMonths,
                    weekdays: this.persianDays,
                    weekdaysShort: this.persianDaysShort
                }
            });
        } else {
            // Fallback: تنظیم ساده
            input.placeholder = 'YYYY/MM/DD';
            input.addEventListener('input', (e) => {
                e.target.value = this.toPersianNumbers(e.target.value);
            });
        }
    }

    /**
     * تنظیم تقویم میلادی
     */
    setupGregorianDatePicker(input) {
        if (typeof Pikaday !== 'undefined') {
            new Pikaday({
                field: input,
                format: 'YYYY-MM-DD'
            });
        }
        
        // تنظیم placeholder
        input.placeholder = this.locale === 'en' ? 'YYYY-MM-DD' : 'YYYY-MM-DD';
    }

    /**
     * رصد تقویم‌های جدید
     */
    observeNewDatePickers() {
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                mutation.addedNodes.forEach((node) => {
                    if (node.nodeType === 1) { // Element node
                        const dateInputs = node.querySelectorAll ? 
                            node.querySelectorAll('input[type="date"], .datepicker, .admin-datepicker') : [];
                        
                        dateInputs.forEach(input => {
                            this.setupDatePicker(input);
                        });
                    }
                });
            });
        });

        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    /**
     * به‌روزرسانی نمودارهای موجود
     */
    updateExistingCharts() {
        // به‌روزرسانی نمودارهای Chart.js
        if (typeof Chart !== 'undefined') {
            // Chart.js v3+ doesn't have Chart.instances
            // We'll update charts when they are created instead
            console.log('Chart.js detected, charts will be updated individually');
        }
    }

    /**
     * به‌روزرسانی برچسب‌های نمودار
     */
    updateChartLabels(chart) {
        if (!chart.data || !chart.data.labels) return;
        
        // به‌روزرسانی برچسب‌ها بر اساس زبان
        chart.data.labels = chart.data.labels.map(label => {
            if (typeof label === 'string' && label.match(/^\d{4}-\d{2}-\d{2}$/)) {
                return this.formatDateForChart(label);
            }
            return this.locale === 'fa' ? this.toPersianNumbers(label) : label;
        });
        
        chart.update();
    }

    /**
     * تبدیل تاریخ از طریق سرور (برای fallback)
     */
    async convertDateViaServer(date, type) {
        try {
            const response = await fetch('/api/convert-date', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content')
                },
                body: JSON.stringify({ date, type, locale: this.locale })
            });
            
            const result = await response.json();
            return result.converted_date || date;
        } catch (error) {
            console.warn('Date conversion failed:', error);
            return date;
        }
    }

    /**
     * دریافت تاریخ امروز بر اساس زبان
     */
    getTodayFormatted() {
        const today = new Date();
        return this.formatDateForChart(today);
    }

    /**
     * اعتبارسنجی تاریخ
     */
    validateDate(dateString) {
        if (this.locale === 'fa') {
            // اعتبارسنجی تاریخ شمسی
            return /^\d{4}\/\d{1,2}\/\d{1,2}$/.test(dateString);
        } else {
            // اعتبارسنجی تاریخ میلادی
            return !isNaN(Date.parse(dateString));
        }
    }
    };
}

// راه‌اندازی خودکار فقط اگر قبلاً راه‌اندازی نشده باشد
if (!window.localeDateManager) {
    document.addEventListener('DOMContentLoaded', () => {
        window.localeDateManager = new window.LocaleDateManager();
    });
}

// Export برای استفاده در ماژول‌ها
if (typeof module !== 'undefined' && module.exports) {
    module.exports = window.LocaleDateManager;
}
