/**
 * Modern Admin Change Password Page JavaScript
 * Enhanced interactions, security features, and animations
 */

class AdvancedPasswordChanger {
    constructor() {
        this.init();
        this.setupEventListeners();
        this.setupSecurityFeatures();
        this.animatePageLoad();
        this.startSecurityMonitoring();
    }

    init() {
        // Form elements
        this.form = document.getElementById('password-form');
        this.submitBtn = document.getElementById('submit-btn');
        this.passwordInput = document.getElementById('password');
        this.confirmInput = document.getElementById('password_confirmation');
        this.currentPasswordInput = document.getElementById('current_password');
        
        // UI elements
        this.strengthFill = document.getElementById('strength-fill');
        this.strengthText = document.getElementById('strength-text');
        this.passwordMatch = document.getElementById('password-match');
        
        // Security elements
        this.securityScore = document.querySelector('.score-text');
        this.securityCircle = document.querySelector('.security-score');
        
        // Requirements
        this.requirements = {
            length: document.getElementById('req-length'),
            uppercase: document.getElementById('req-uppercase'),
            lowercase: document.getElementById('req-lowercase'),
            number: document.getElementById('req-number'),
            special: document.getElementById('req-special')
        };

        // State
        this.passwordStrength = 0;
        this.isFormValid = false;
        this.securityLevel = 'low';
    }

    setupEventListeners() {
        // Password toggle functionality
        document.querySelectorAll('.password-toggle').forEach(toggle => {
            toggle.addEventListener('click', (e) => this.togglePassword(e));
        });

        // Password strength checking
        if (this.passwordInput) {
            this.passwordInput.addEventListener('input', (e) => {
                this.checkPasswordStrength(e.target.value);
                this.checkPasswordMatch();
                this.updateFormValidation();
            });
        }

        // Password confirmation checking
        if (this.confirmInput) {
            this.confirmInput.addEventListener('input', () => {
                this.checkPasswordMatch();
                this.updateFormValidation();
            });
        }

        // Current password validation
        if (this.currentPasswordInput) {
            this.currentPasswordInput.addEventListener('input', () => {
                this.updateFormValidation();
            });
        }

        // Form submission
        if (this.form) {
            this.form.addEventListener('submit', (e) => this.handleSubmit(e));
        }

        // Enhanced input interactions
        document.querySelectorAll('.form-control').forEach(input => {
            input.addEventListener('focus', (e) => this.handleInputFocus(e));
            input.addEventListener('blur', (e) => this.handleInputBlur(e));
            input.addEventListener('keydown', (e) => this.handleKeyDown(e));
        });

        // Security dashboard interactions
        document.querySelectorAll('.stat-item').forEach(item => {
            item.addEventListener('click', (e) => this.handleStatClick(e));
        });
    }

    setupSecurityFeatures() {
        // Password generation
        this.setupPasswordGenerator();
        
        // Security tips interaction
        this.setupSecurityTips();
        
        // Caps lock detection
        this.setupCapsLockDetection();
        
        // Copy prevention
        this.setupCopyPrevention();
    }

    togglePassword(e) {
        const button = e.currentTarget;
        const targetId = button.getAttribute('data-target');
        const input = document.getElementById(targetId);
        const icon = button.querySelector('i');

        if (input.type === 'password') {
            input.type = 'text';
            icon.className = 'fas fa-eye-slash';
            button.setAttribute('aria-label', 'مخفی کردن رمز عبور');
        } else {
            input.type = 'password';
            icon.className = 'fas fa-eye';
            button.setAttribute('aria-label', 'نمایش رمز عبور');
        }

        // Enhanced animation
        button.style.transform = 'scale(0.9) rotate(180deg)';
        setTimeout(() => {
            button.style.transform = 'scale(1) rotate(0deg)';
        }, 200);
    }

    checkPasswordStrength(password) {
        let score = 0;
        const checks = {
            length: password.length >= 8,
            uppercase: /[A-Z]/.test(password),
            lowercase: /[a-z]/.test(password),
            number: /[0-9]/.test(password),
            special: /[^A-Za-z0-9]/.test(password)
        };

        // Advanced scoring
        if (password.length >= 12) score += 10; // Bonus for longer passwords
        if (password.length >= 16) score += 10; // Extra bonus
        if (/[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]/.test(password)) score += 5; // Special chars bonus

        // Update requirement indicators with enhanced animations
        Object.keys(checks).forEach(key => {
            const element = this.requirements[key];
            const icon = element.querySelector('i');
            
            if (checks[key]) {
                element.classList.add('valid');
                icon.className = 'fas fa-check';
                score += 20;
                
                // Add success animation
                element.style.animation = 'checkmarkBounce 0.5s ease-out';
                setTimeout(() => {
                    element.style.animation = '';
                }, 500);
            } else {
                element.classList.remove('valid');
                icon.className = 'fas fa-times';
            }
        });

        this.passwordStrength = Math.min(score, 100);
        this.updateStrengthBar(this.passwordStrength);
        this.updateSecurityLevel();
    }

    updateStrengthBar(score) {
        const percentage = Math.min(score, 100);
        this.strengthFill.style.width = percentage + '%';

        let color, text, level;
        if (score < 40) {
            color = '#ef4444';
            text = 'ضعیف';
            level = 'weak';
        } else if (score < 70) {
            color = '#fbbf24';
            text = 'متوسط';
            level = 'medium';
        } else if (score < 90) {
            color = '#4ade80';
            text = 'قوی';
            level = 'strong';
        } else {
            color = '#10b981';
            text = 'بسیار قوی';
            level = 'very-strong';
        }

        this.strengthFill.style.background = color;
        this.strengthText.textContent = `قدرت رمز عبور: ${text}`;
        this.strengthText.style.color = color;
        
        // Add glow effect
        this.strengthFill.style.boxShadow = `0 0 10px ${color}`;
        
        this.securityLevel = level;
    }

    updateSecurityLevel() {
        // Update security circle class
        this.securityCircle.className = 'security-score';
        
        if (this.passwordStrength >= 80) {
            this.securityCircle.classList.add('high-security');
        } else if (this.passwordStrength >= 60) {
            this.securityCircle.classList.add('medium-security');
        } else {
            this.securityCircle.classList.add('low-security');
        }
    }

    checkPasswordMatch() {
        const password = this.passwordInput.value;
        const confirm = this.confirmInput.value;

        if (confirm === '') {
            this.passwordMatch.textContent = '';
            this.passwordMatch.className = 'password-match';
            return false;
        }

        if (password === confirm) {
            this.passwordMatch.innerHTML = '<i class="fas fa-check"></i> رمز عبور مطابقت دارد';
            this.passwordMatch.className = 'password-match match';
            return true;
        } else {
            this.passwordMatch.innerHTML = '<i class="fas fa-times"></i> رمز عبور مطابقت ندارد';
            this.passwordMatch.className = 'password-match no-match';
            return false;
        }
    }

    updateFormValidation() {
        const currentPassword = this.currentPasswordInput.value;
        const newPassword = this.passwordInput.value;
        const confirmPassword = this.confirmInput.value;
        
        const isValid = currentPassword.length > 0 && 
                       newPassword.length >= 8 && 
                       newPassword === confirmPassword &&
                       this.passwordStrength >= 60;

        this.isFormValid = isValid;
        this.submitBtn.disabled = !isValid;
        
        if (isValid) {
            this.submitBtn.classList.add('valid');
        } else {
            this.submitBtn.classList.remove('valid');
        }
    }

    handleInputFocus(e) {
        const wrapper = e.target.closest('.input-wrapper');
        if (wrapper) {
            wrapper.style.transform = 'scale(1.02)';
            wrapper.style.transition = 'transform 0.2s ease';
            wrapper.classList.add('focused');
        }

        // Add focus ring
        e.target.style.boxShadow = '0 0 0 3px rgba(102, 126, 234, 0.2)';
    }

    handleInputBlur(e) {
        const wrapper = e.target.closest('.input-wrapper');
        if (wrapper) {
            wrapper.style.transform = 'scale(1)';
            wrapper.classList.remove('focused');
        }

        e.target.style.boxShadow = '';
    }

    handleKeyDown(e) {
        // Enhanced keyboard navigation
        if (e.key === 'Enter') {
            const inputs = Array.from(document.querySelectorAll('.form-control'));
            const currentIndex = inputs.indexOf(e.target);
            const nextInput = inputs[currentIndex + 1];
            
            if (nextInput) {
                e.preventDefault();
                nextInput.focus();
            } else if (this.isFormValid) {
                e.preventDefault();
                this.submitBtn.click();
            }
        }
    }

    handleSubmit(e) {
        if (!this.isFormValid) {
            e.preventDefault();
            this.showError('لطفاً تمام فیلدها را به درستی پر کنید');
            return;
        }

        // Add loading state
        this.submitBtn.classList.add('loading');
        this.submitBtn.disabled = true;

        // Add success animation
        this.addSubmitAnimation();
        
        // Log security event
        this.logSecurityEvent('password_change_attempt');
    }

    setupPasswordGenerator() {
        // Add password generator button (optional feature)
        const generateBtn = document.createElement('button');
        generateBtn.type = 'button';
        generateBtn.className = 'btn btn-outline-secondary btn-sm';
        generateBtn.innerHTML = '<i class="fas fa-magic"></i> تولید رمز عبور';
        generateBtn.addEventListener('click', () => this.generateSecurePassword());
        
        // Add to password field
        const passwordGroup = this.passwordInput.closest('.form-group');
        if (passwordGroup) {
            passwordGroup.appendChild(generateBtn);
        }
    }

    generateSecurePassword() {
        const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*';
        let password = '';
        
        // Ensure at least one of each type
        password += 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'[Math.floor(Math.random() * 26)];
        password += 'abcdefghijklmnopqrstuvwxyz'[Math.floor(Math.random() * 26)];
        password += '0123456789'[Math.floor(Math.random() * 10)];
        password += '!@#$%^&*'[Math.floor(Math.random() * 8)];
        
        // Fill the rest
        for (let i = 4; i < 12; i++) {
            password += chars[Math.floor(Math.random() * chars.length)];
        }
        
        // Shuffle
        password = password.split('').sort(() => Math.random() - 0.5).join('');
        
        this.passwordInput.value = password;
        this.confirmInput.value = password;
        
        this.checkPasswordStrength(password);
        this.checkPasswordMatch();
        this.updateFormValidation();
        
        // Show notification
        this.showSuccess('رمز عبور قوی تولید شد');
    }

    setupSecurityTips() {
        document.querySelectorAll('.tip-item').forEach(tip => {
            tip.addEventListener('click', () => {
                tip.style.animation = 'pulse 0.5s ease-out';
                setTimeout(() => {
                    tip.style.animation = '';
                }, 500);
            });
        });
    }

    setupCapsLockDetection() {
        document.querySelectorAll('input[type="password"]').forEach(input => {
            input.addEventListener('keydown', (e) => {
                if (e.getModifierState && e.getModifierState('CapsLock')) {
                    this.showCapsLockWarning(input);
                } else {
                    this.hideCapsLockWarning(input);
                }
            });
        });
    }

    showCapsLockWarning(input) {
        let warning = input.parentElement.querySelector('.caps-lock-warning');
        if (!warning) {
            warning = document.createElement('div');
            warning.className = 'caps-lock-warning';
            warning.innerHTML = '<i class="fas fa-exclamation-triangle"></i> Caps Lock فعال است';
            warning.style.cssText = `
                color: #fbbf24;
                font-size: 0.8rem;
                margin-top: 0.25rem;
                display: flex;
                align-items: center;
                gap: 0.5rem;
            `;
            input.parentElement.appendChild(warning);
        }
    }

    hideCapsLockWarning(input) {
        const warning = input.parentElement.querySelector('.caps-lock-warning');
        if (warning) {
            warning.remove();
        }
    }

    setupCopyPrevention() {
        // Prevent copying of password fields (optional security feature)
        document.querySelectorAll('input[type="password"]').forEach(input => {
            input.addEventListener('copy', (e) => {
                e.preventDefault();
                this.showWarning('کپی کردن رمز عبور از نظر امنیتی توصیه نمی‌شود');
            });
        });
    }

    startSecurityMonitoring() {
        // Monitor for suspicious activity
        let failedAttempts = 0;
        const maxAttempts = 3;
        
        this.form.addEventListener('submit', (e) => {
            if (!this.isFormValid) {
                failedAttempts++;
                if (failedAttempts >= maxAttempts) {
                    this.showError('تعداد تلاش‌های ناموفق زیاد است. لطفاً کمی صبر کنید.');
                    this.submitBtn.disabled = true;
                    setTimeout(() => {
                        this.submitBtn.disabled = false;
                        failedAttempts = 0;
                    }, 30000); // 30 second cooldown
                }
            }
        });
    }

    handleStatClick(e) {
        const statItem = e.currentTarget;
        const statValue = statItem.querySelector('.stat-value');
        const statLabel = statItem.querySelector('.stat-label');
        
        // Add click animation
        statItem.style.transform = 'scale(0.95)';
        setTimeout(() => {
            statItem.style.transform = 'scale(1)';
        }, 150);
        
        // Show detailed info (could be expanded)
        this.showInfo(`${statLabel.textContent}: ${statValue.textContent}`);
    }

    animatePageLoad() {
        // Enhanced page load animations
        setTimeout(() => {
            const scoreCircle = document.querySelector('.score-fill');
            if (scoreCircle) {
                scoreCircle.style.strokeDasharray = scoreCircle.getAttribute('stroke-dasharray');
            }
        }, 500);

        // Staggered animations for stats
        const statItems = document.querySelectorAll('.stat-item');
        statItems.forEach((item, index) => {
            item.style.opacity = '0';
            item.style.transform = 'translateY(20px)';
            
            setTimeout(() => {
                item.style.transition = 'all 0.6s cubic-bezier(0.4, 0, 0.2, 1)';
                item.style.opacity = '1';
                item.style.transform = 'translateY(0)';
            }, 200 + (index * 100));
        });

        // Form elements animation
        const formGroups = document.querySelectorAll('.form-group');
        formGroups.forEach((group, index) => {
            group.style.opacity = '0';
            group.style.transform = 'translateX(20px)';
            
            setTimeout(() => {
                group.style.transition = 'all 0.6s cubic-bezier(0.4, 0, 0.2, 1)';
                group.style.opacity = '1';
                group.style.transform = 'translateX(0)';
            }, 400 + (index * 150));
        });
    }

    addSubmitAnimation() {
        // Enhanced submit animation
        const ripple = document.createElement('span');
        ripple.style.cssText = `
            position: absolute;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.6);
            transform: scale(0);
            animation: ripple 0.6s linear;
            pointer-events: none;
            width: 100px;
            height: 100px;
            left: 50%;
            top: 50%;
            margin-left: -50px;
            margin-top: -50px;
        `;
        
        this.submitBtn.appendChild(ripple);
        
        setTimeout(() => {
            ripple.remove();
        }, 600);
    }

    logSecurityEvent(event) {
        // Log security events (could be sent to server)
        console.log(`Security Event: ${event} at ${new Date().toISOString()}`);
    }

    showSuccess(message) {
        this.showNotification(message, 'success');
    }

    showError(message) {
        this.showNotification(message, 'error');
    }

    showWarning(message) {
        this.showNotification(message, 'warning');
    }

    showInfo(message) {
        this.showNotification(message, 'info');
    }

    showNotification(message, type) {
        const notification = document.createElement('div');
        notification.className = `alert alert-${type === 'error' ? 'danger' : type}`;
        notification.innerHTML = `<i class="fas fa-${type === 'success' ? 'check' : type === 'error' ? 'exclamation' : 'info'}-circle"></i> ${message}`;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.remove();
        }, 5000);
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new AdvancedPasswordChanger();
    console.log('🔐 Advanced Password Changer initialized');
});
