<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\View;
use App\Services\MenuService;
use App\Helpers\BreadcrumbHelper;

class MenuServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        $this->app->singleton(MenuService::class, function ($app) {
            return new MenuService();
        });
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Share menu and header data with all views
        View::composer('*', function ($view) {
            $menuService = app(MenuService::class);
            $currentLanguage = app()->getLocale() ?? 'fa';
            
            try {
                // Get menu and header data
                $headerData = $menuService->getHeaderData($currentLanguage);
                
                // Share data with all views
                $view->with([
                    'site_menus' => $headerData['menus'],
                    'header_settings' => $headerData['header_settings'],
                    'user_buttons' => $headerData['buttons'],
                    'is_guest_user' => $headerData['is_guest'],
                    'current_auth_user' => $headerData['current_user'],
                    'menu_service' => $menuService,
                ]);
                
                // Generate breadcrumb
                $breadcrumb = BreadcrumbHelper::generate();
                $pageTitle = BreadcrumbHelper::getPageTitle($breadcrumb);

                // Additional helper variables
                $view->with([
                    'show_search_box' => $menuService->shouldShowSearchBox($currentLanguage),
                    'search_placeholder' => $menuService->getSearchPlaceholder($currentLanguage),
                    'show_language_switcher' => $menuService->shouldShowLanguageSwitcher($currentLanguage),
                    'show_user_menu' => $menuService->shouldShowUserMenu($currentLanguage),
                    'header_style' => $menuService->getHeaderStyle($currentLanguage),
                    'is_sticky_header' => $menuService->isStickyHeader($currentLanguage),
                    'show_breadcrumb' => $menuService->shouldShowBreadcrumb($currentLanguage),
                    'breadcrumb' => $breadcrumb,
                    'title' => $pageTitle,
                ]);
                
            } catch (\Exception $e) {
                // Fallback data in case of errors
                $view->with([
                    'site_menus' => collect([]),
                    'header_settings' => [],
                    'user_buttons' => collect([]),
                    'is_guest_user' => true,
                    'current_auth_user' => null,
                    'show_search_box' => true,
                    'search_placeholder' => 'جستجو...',
                    'show_language_switcher' => true,
                    'show_user_menu' => true,
                    'header_style' => 'default',
                    'is_sticky_header' => true,
                    'show_breadcrumb' => true,
                    'breadcrumb' => [],
                    'title' => 'صفحه فعلی',
                ]);
                
                // Log the error for debugging
                \Log::error('MenuServiceProvider error: ' . $e->getMessage());
            }
        });
    }
}
