<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class RatingStatistics extends Model
{
    protected $table = 'rating_statistics';

    protected $fillable = [
        'title_id',
        'total_ratings',
        'average_rating',
        'rating_1_count',
        'rating_2_count',
        'rating_3_count',
        'rating_4_count',
        'rating_5_count',
        'service_avg',
        'quality_avg',
        'location_avg',
        'price_avg',
        'cleanliness_avg',
        'total_comments',
        'last_rating_date',
        'last_comment_date'
    ];

    protected $casts = [
        'total_ratings' => 'integer',
        'average_rating' => 'decimal:2',
        'rating_1_count' => 'integer',
        'rating_2_count' => 'integer',
        'rating_3_count' => 'integer',
        'rating_4_count' => 'integer',
        'rating_5_count' => 'integer',
        'service_avg' => 'decimal:2',
        'quality_avg' => 'decimal:2',
        'location_avg' => 'decimal:2',
        'price_avg' => 'decimal:2',
        'cleanliness_avg' => 'decimal:2',
        'total_comments' => 'integer',
        'last_rating_date' => 'datetime',
        'last_comment_date' => 'datetime',
        'updated_at' => 'datetime'
    ];

    protected $dates = [
        'last_rating_date',
        'last_comment_date',
        'updated_at'
    ];

    // Relationships
    public function location()
    {
        return $this->belongsTo(Location::class, 'title_id', 'title_id');
    }

    // Accessors
    public function getStarRatingAttribute()
    {
        $rating = $this->average_rating ?? 0;
        $stars = '';
        for ($i = 1; $i <= 5; $i++) {
            $stars .= $i <= $rating ? '★' : '☆';
        }
        return $stars;
    }

    public function getRatingDistributionAttribute()
    {
        return [
            1 => $this->rating_1_count,
            2 => $this->rating_2_count,
            3 => $this->rating_3_count,
            4 => $this->rating_4_count,
            5 => $this->rating_5_count,
        ];
    }

    public function getRatingPercentagesAttribute()
    {
        $total = $this->total_ratings;
        if ($total == 0) {
            return [1 => 0, 2 => 0, 3 => 0, 4 => 0, 5 => 0];
        }

        return [
            1 => round(($this->rating_1_count / $total) * 100, 1),
            2 => round(($this->rating_2_count / $total) * 100, 1),
            3 => round(($this->rating_3_count / $total) * 100, 1),
            4 => round(($this->rating_4_count / $total) * 100, 1),
            5 => round(($this->rating_5_count / $total) * 100, 1),
        ];
    }

    public function getDetailedAveragesAttribute()
    {
        return [
            'service' => $this->service_avg,
            'quality' => $this->quality_avg,
            'location' => $this->location_avg,
            'price' => $this->price_avg,
            'cleanliness' => $this->cleanliness_avg,
        ];
    }

    public function getOverallScoreAttribute()
    {
        $averages = array_filter($this->detailed_averages);
        return count($averages) > 0 ? round(array_sum($averages) / count($averages), 1) : $this->average_rating;
    }

    public function getQualityLevelAttribute()
    {
        $rating = $this->average_rating;

        if ($rating >= 4.5) return 'عالی';
        if ($rating >= 4.0) return 'خیلی خوب';
        if ($rating >= 3.5) return 'خوب';
        if ($rating >= 3.0) return 'متوسط';
        if ($rating >= 2.0) return 'ضعیف';
        return 'بسیار ضعیف';
    }

    public function getQualityColorAttribute()
    {
        $rating = $this->average_rating;

        if ($rating >= 4.5) return '#28a745'; // Green
        if ($rating >= 4.0) return '#20c997'; // Teal
        if ($rating >= 3.5) return '#ffc107'; // Yellow
        if ($rating >= 3.0) return '#fd7e14'; // Orange
        if ($rating >= 2.0) return '#dc3545'; // Red
        return '#6c757d'; // Gray
    }

    // Static methods
    public static function updateForLocation($titleId)
    {
        $ratings = LocationRating::approved()->forLocation($titleId)->get();

        if ($ratings->isEmpty()) {
            static::where('title_id', $titleId)->delete();
            return null;
        }

        $totalRatings = $ratings->count();
        $averageRating = $ratings->avg('rating');

        $ratingCounts = [
            'rating_1_count' => $ratings->where('rating', 1)->count(),
            'rating_2_count' => $ratings->where('rating', 2)->count(),
            'rating_3_count' => $ratings->where('rating', 3)->count(),
            'rating_4_count' => $ratings->where('rating', 4)->count(),
            'rating_5_count' => $ratings->where('rating', 5)->count(),
        ];

        $detailedAverages = [
            'service_avg' => $ratings->whereNotNull('service_rating')->avg('service_rating'),
            'quality_avg' => $ratings->whereNotNull('quality_rating')->avg('quality_rating'),
            'location_avg' => $ratings->whereNotNull('location_rating')->avg('location_rating'),
            'price_avg' => $ratings->whereNotNull('price_rating')->avg('price_rating'),
            'cleanliness_avg' => $ratings->whereNotNull('cleanliness_rating')->avg('cleanliness_rating'),
        ];

        $lastRatingDate = $ratings->max('created_at');
        $totalComments = $ratings->whereNotNull('comment')->count();

        return static::updateOrCreate(
            ['title_id' => $titleId],
            array_merge([
                'total_ratings' => $totalRatings,
                'average_rating' => round($averageRating, 2),
                'total_comments' => $totalComments,
                'last_rating_date' => $lastRatingDate,
                'updated_at' => now()
            ], $ratingCounts, $detailedAverages)
        );
    }

    public static function getTopRatedLocations($limit = 10)
    {
        return static::with('location')
            ->where('total_ratings', '>=', 5) // At least 5 ratings
            ->orderBy('average_rating', 'desc')
            ->orderBy('total_ratings', 'desc')
            ->limit($limit)
            ->get();
    }

    public static function getMostReviewedLocations($limit = 10)
    {
        return static::with('location')
            ->orderBy('total_ratings', 'desc')
            ->orderBy('average_rating', 'desc')
            ->limit($limit)
            ->get();
    }

    public static function getRecentlyRatedLocations($limit = 10)
    {
        return static::with('location')
            ->whereNotNull('last_rating_date')
            ->orderBy('last_rating_date', 'desc')
            ->limit($limit)
            ->get();
    }

    // Helper methods
    public function hasEnoughRatings($minimum = 5)
    {
        return $this->total_ratings >= $minimum;
    }

    public function isHighlyRated($threshold = 4.0)
    {
        return $this->average_rating >= $threshold;
    }

    public function isPopular($minimumRatings = 20)
    {
        return $this->total_ratings >= $minimumRatings;
    }

    public function needsMoreReviews($target = 10)
    {
        return $this->total_ratings < $target;
    }

    public function getRecommendationScore()
    {
        // Calculate a score based on rating and number of reviews
        $ratingScore = ($this->average_rating / 5) * 70; // 70% weight for rating
        $reviewScore = min(($this->total_ratings / 50) * 30, 30); // 30% weight for review count (capped at 50 reviews)

        return round($ratingScore + $reviewScore, 1);
    }
}
