<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class DisplayZone extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'ad_display_zones';

    protected $fillable = [
        'zone_key',
        'zone_name',
        'description',
        'position',
        'max_desktop_cards',
        'max_mobile_cards',
        'card_layout',
        'is_active',
        'display_order',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'max_desktop_cards' => 'integer',
        'max_mobile_cards' => 'integer',
        'display_order' => 'integer',
    ];

    /**
     * رابطه با آگهی‌ها
     */
    public function ads()
    {
        return $this->belongsToMany(Ad::class, 'zone_ads', 'zone_id', 'ad_id')
            ->withPivot('location_id', 'display_order', 'is_active', 'start_date', 'end_date')
            ->withTimestamps();
    }

    /**
     * رابطه با تنظیمات آگهی در این زون
     */
    public function adSettings()
    {
        return $this->hasMany(ZoneAd::class, 'zone_id');
    }

    /**
     * رابطه با کاربر ایجاد کننده
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * رابطه با کاربر ویرایش کننده
     */
    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * فیلتر کردن زون‌های فعال
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * دریافت آگهی‌های فعال برای این زون
     */
    public function getActiveAds($locationId = null)
    {
        $query = $this->ads()
            ->wherePivot('is_active', true)
            ->where('ads.is_active', true)
            ->whereRaw('(zone_ads.start_date IS NULL OR zone_ads.start_date <= NOW())')
            ->whereRaw('(zone_ads.end_date IS NULL OR zone_ads.end_date >= NOW())');

        if ($locationId) {
            $query->wherePivot('location_id', $locationId);
        }

        return $query->orderBy('zone_ads.display_order')
            ->take($this->max_desktop_cards ?? 10)
            ->get();
    }
}
