<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\City;
use App\Models\Country;
use App\Models\County;
use App\Models\District;
use App\Models\Location;
use App\Models\Province;
use App\Models\Zone;
use App\Services\AdService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class HomeController extends Controller
{
    protected $adService;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(AdService $adService)
    {
        $this->adService = $adService;
    }

    /**
     * Show the application homepage.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function index()
    {
        // Get all parent categories with their children
        $categories = Category::where('parent_id', null)
            ->where('show_in_filters', true)
            ->with([
                'children' => function ($query) {
                    $query->where('show_in_filters', true)
                        ->orderBy('sort_order')
                        ->with('translations');
                },
                'translations'
            ])
            ->orderBy('sort_order')
            ->get();

        // Get all active locations with their categories
        $locations = Location::with(['categoryRelation.activeTemplateRelation', 'city', 'province'])
            ->where('is_active', true)
            ->get();

        // Get featured locations based on package type
        $featuredLocations = Location::whereIn('package_type', ['diamond', 'gold', 'silver', 'bronze'])
            ->where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('package_expires_at')
                    ->orWhere('package_expires_at', '>=', now());
            })
            ->orderByRaw("FIELD(package_type, 'diamond', 'gold', 'silver', 'bronze')")
            ->get();

        // Get diamond locations
        $diamondLocations = Location::where('package_type', 'diamond')
            ->where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('package_expires_at')
                    ->orWhere('package_expires_at', '>=', now());
            })
            ->get();

        // Get diamond ads for homepage
        $diamondAds = $this->adService->getHomepageDiamondAds(4);

        // For each category, calculate the number of locations
        foreach ($categories as $category) {
            $category->location_count = $category->getTotalLocationCount();

            foreach ($category->children as $child) {
                $child->location_count = $child->locations()->count();
            }
        }

        // Get active countries with location count, ordered by count
        $countries = $this->getActiveCountriesWithCount();

        return view('welcome', compact('categories', 'locations', 'countries', 'featuredLocations', 'diamondLocations', 'diamondAds'));
    }

    /**
     * Handle Ajax search requests
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function search(Request $request)
    {
        try {
            // Log the request parameters
            \Log::info('Search request parameters:', [
                'all' => $request->all(),
                'query' => $request->query(),
            ]);

            $query = Location::with(['categoryRelation', 'city', 'province', 'country', 'county', 'district', 'village', 'zone'])
                ->where('is_active', true);

            // Log the base query
            \Log::info('Base query:', [
                'sql' => $query->toSql(),
                'bindings' => $query->getBindings(),
            ]);

            // Apply search query if provided
            if ($request->has('q') && ! empty($request->q)) {
                $searchTerm = $request->q;
                $query->where(function ($q) use ($searchTerm) {
                    $q->where('title', 'like', "%{$searchTerm}%")
                        ->orWhere('description', 'like', "%{$searchTerm}%")
                        ->orWhere('tags', 'like', "%{$searchTerm}%")
                        ->orWhere('address', 'like', "%{$searchTerm}%");
                });

                \Log::info('After search term filter:', [
                    'search_term' => $searchTerm,
                    'sql' => $query->toSql(),
                    'bindings' => $query->getBindings(),
                ]);
            }

            // Apply category filter if provided
            if ($request->has('category') && ! empty($request->category)) {
                $categoryIds = explode(',', $request->category);

                // If parent categories are selected, include children
                $allCategoryIds = $categoryIds;

                try {
                    foreach ($categoryIds as $catId) {
                        $category = Category::find($catId);
                        if ($category && $category->parent_id === null) {
                            // Add all children of this parent category
                            $childIds = [];

                            // Get direct children first
                            $directChildren = $category->children()->pluck('id')->toArray();
                            $childIds = array_merge($childIds, $directChildren);

                            $allCategoryIds = array_merge($allCategoryIds, $childIds);
                        }
                    }
                } catch (\Exception $e) {
                    \Log::error('Error while getting child categories: '.$e->getMessage());
                    // Continue with just the explicitly selected categories
                }

                // Filter categories to ensure they're all valid integers
                $allCategoryIds = array_filter(array_unique($allCategoryIds), function ($id) {
                    return is_numeric($id) && $id > 0;
                });

                if (! empty($allCategoryIds)) {
                    $query->whereIn('category_id', $allCategoryIds);
                }

                \Log::info('After category filter:', [
                    'category_ids' => $categoryIds,
                    'all_category_ids' => $allCategoryIds,
                    'sql' => $query->toSql(),
                    'bindings' => $query->getBindings(),
                ]);
            }

            // Apply geographic filters
            if ($request->has('country') && ! empty($request->country)) {
                $query->where('country_id', $request->country);

                \Log::info('After country filter:', [
                    'country_id' => $request->country,
                    'sql' => $query->toSql(),
                    'bindings' => $query->getBindings(),
                ]);

                if ($request->has('province') && ! empty($request->province)) {
                    $query->where('province_id', $request->province);

                    \Log::info('After province filter:', [
                        'province_id' => $request->province,
                        'sql' => $query->toSql(),
                        'bindings' => $query->getBindings(),
                    ]);

                    if ($request->has('county') && ! empty($request->county)) {
                        $query->where('county_id', $request->county);

                        if ($request->has('district') && ! empty($request->district)) {
                            $query->where('district_id', $request->district);

                            if ($request->has('city') && ! empty($request->city)) {
                                $query->where('city_id', $request->city);
                            }

                            if ($request->has('village') && ! empty($request->village)) {
                                $query->where('village_id', $request->village);
                            }
                        }
                    }
                }

                if ($request->has('zone') && ! empty($request->zone)) {
                    $query->where('zone_id', $request->zone);
                }
            }

            // Check if we have any locations before continuing
            $countCheck = clone $query;
            $count = $countCheck->count();
            \Log::info('Final query count before serialization:', [
                'count' => $count,
                'sql' => $query->toSql(),
                'bindings' => $query->getBindings(),
            ]);

            // If get_categories_only is set, return categories from filtered results
            if ($request->has('get_categories_only') && $request->get_categories_only == '1') {
                // Get all locations that match the filters
                $locationIds = $query->pluck('id')->toArray();

                // Debug: Log location count
                \Log::info('Filtered locations count: '.count($locationIds));

                // If there are no locations, return empty categories array
                if (empty($locationIds)) {
                    return response()->json([
                        'success' => true,
                        'count' => 0,
                        'categories' => [],
                    ]);
                }

                // Debug: Log query for categories
                \Log::info('Building category query with '.count($locationIds).' location IDs');

                // Get all parent categories with their children that have locations matching the filters
                $categories = Category::whereNull('parent_id')
                    ->where('show_in_filters', true)
                    ->orderBy('sort_order')
                    ->with([
                        'children' => function ($query) use ($locationIds) {
                            $query->where('show_in_filters', true)
                                ->orderBy('sort_order')
                                ->with(['translations'])
                                ->withCount(['locations' => function ($q) use ($locationIds) {
                                    $q->whereIn('id', $locationIds)
                                        ->where('is_active', true);
                                }]);
                        },
                        'children.translations',
                        'translations'
                    ])
                    ->withCount(['locations' => function ($query) use ($locationIds) {
                        $query->whereIn('id', $locationIds)
                            ->where('is_active', true);
                    }])
                    ->get();

                // Debug: Log categories found
                \Log::info('Found '.$categories->count().' parent categories');

                // Map to the expected format
                $formattedCategories = $categories->map(function ($category) {
                    // Debug: Log category counts
                    \Log::info("Category {$category->id} ({$category->category_fa}) has {$category->locations_count} locations");

                    // Calculate the total count including children
                    $totalCount = $category->locations_count;

                    // Get children with their counts
                    $children = $category->children->map(function ($child) {
                        // Debug: Log child category counts
                        \Log::info("  Child {$child->id} ({$child->category_fa}) has {$child->locations_count} locations");

                        return [
                            'id' => $child->id,
                            'category_fa' => $child->category_fa,
                            'emoji' => $child->emoji,
                            'location_count' => $child->locations_count,
                        ];
                    });

                    // If parent has zero direct locations, add up children's counts
                    if ($totalCount == 0) {
                        foreach ($children as $child) {
                            $totalCount += $child['location_count'];
                        }
                        \Log::info("  Total count for category {$category->id} including children: {$totalCount}");
                    }

                    return [
                        'id' => $category->id,
                        'category_fa' => $category->category_fa,
                        'emoji' => $category->emoji,
                        'location_count' => $totalCount,
                        'children' => $children,
                    ];
                });

                // Include the original and formatted data in response for debugging
                $response = [
                    'success' => true,
                    'count' => count($locationIds),
                    'categories' => $formattedCategories,
                    'debug' => [
                        'location_ids_count' => count($locationIds),
                        'parent_categories_count' => $categories->count(),
                    ],
                ];

                return response()->json($response, 200, [], JSON_UNESCAPED_UNICODE | JSON_PARTIAL_OUTPUT_ON_ERROR);
            }

            try {
                $locations = $query->get();

                \Log::info('Final result:', [
                    'count' => $locations->count(),
                    'first_few_ids' => $locations->take(5)->pluck('id')->toArray(),
                ]);

                return response()->json([
                    'success' => true,
                    'count' => $locations->count(),
                    'data' => $locations,
                    'debug_info' => [
                        'request_params' => $request->all(),
                        'filter_count' => $count,
                        'has_data' => $locations->isNotEmpty(),
                    ],
                ], 200, [], JSON_UNESCAPED_UNICODE | JSON_PARTIAL_OUTPUT_ON_ERROR);
            } catch (\Exception $e) {
                \Log::error('Error during location data serialization: '.$e->getMessage());

                // Try returning just the basic data without relationships
                try {
                    $basicLocationData = $query->select(['id', 'title', 'category_id', 'category_fa', 'lat', 'lng', 'address', 'tags'])->get();

                    return response()->json([
                        'success' => true,
                        'count' => $basicLocationData->count(),
                        'data' => $basicLocationData,
                        'note' => 'Limited data returned due to serialization issues',
                    ], 200, [], JSON_UNESCAPED_UNICODE | JSON_PARTIAL_OUTPUT_ON_ERROR);
                } catch (\Exception $innerEx) {
                    \Log::error('Error with simplified data retrieval: '.$innerEx->getMessage());
                    throw $innerEx; // Let the outer catch block handle it
                }
            }
        } catch (\Exception $e) {
            // Log the error
            \Log::error('Search error: '.$e->getMessage());
            \Log::error('Exception trace: '.$e->getTraceAsString());

            // Return a simplified error response
            return response()->json([
                'success' => false,
                'error' => 'An error occurred while searching. Please try again.',
                'error_details' => config('app.debug') ? $e->getMessage() : null,
            ], 200);
        }
    }

    /**
     * Handle search suggestions requests
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function searchSuggestions(Request $request)
    {
        try {
            $query = $request->get('q', '');

            if (empty($query) || strlen($query) < 2) {
                return response()->json([
                    'success' => true,
                    'suggestions' => []
                ]);
            }

            $suggestions = [];

            // Search in location titles
            $locations = Location::where('is_active', true)
                ->where('title', 'like', "%{$query}%")
                ->select('id', 'title', 'category_id')
                ->with('category:id,category_fa,emoji')
                ->limit(5)
                ->get();

            foreach ($locations as $location) {
                $suggestions[] = [
                    'type' => 'location',
                    'title' => $location->title,
                    'subtitle' => $location->category ? $location->category->category_fa : '',
                    'icon' => $location->category ? $location->category->emoji : '📍',
                    'url' => route('location.show', $location->id)
                ];
            }

            // Search in categories
            $categories = Category::where('show_in_filters', true)
                ->where(function($q) use ($query) {
                    $q->where('category_fa', 'like', "%{$query}%")
                      ->orWhere('category_en', 'like', "%{$query}%");
                })
                ->select('id', 'category_fa', 'emoji')
                ->limit(3)
                ->get();

            foreach ($categories as $category) {
                $suggestions[] = [
                    'type' => 'category',
                    'title' => $category->category_fa,
                    'subtitle' => 'دسته‌بندی',
                    'icon' => $category->emoji ?: '🏷️',
                    'category_id' => $category->id
                ];
            }

            return response()->json([
                'success' => true,
                'suggestions' => $suggestions
            ]);

        } catch (\Exception $e) {
            \Log::error('Search suggestions error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'suggestions' => []
            ]);
        }
    }

    /**
     * Get active countries with location count, ordered by count
     *
     * @return \Illuminate\Support\Collection
     */
    private function getActiveCountriesWithCount()
    {
        $countries = DB::table('state1_countries')
            ->select(
                'state1_countries.id',
                'state1_countries.name as name_fa',
                'state1_countries.name_en',
                'state1_countries.name_ar',
                'state1_countries.iso_code',
                DB::raw('(SELECT COUNT(*) FROM locations WHERE locations.country_id = state1_countries.id AND locations.is_active = 1) as location_count')
            )
            ->where('state1_countries.is_active', 1)
            ->orderByDesc('location_count')
            ->orderBy(DB::raw("COALESCE(state1_countries.name, state1_countries.name_en, state1_countries.name_ar)"))
            ->get();

        \Log::info('Countries query result:', [
            'sql' => DB::table('state1_countries')
                ->select(
                    'state1_countries.id',
                    'state1_countries.name as name_fa',
                    'state1_countries.name_en',
                    'state1_countries.name_ar',
                    'state1_countries.iso_code',
                    DB::raw('(SELECT COUNT(*) FROM locations WHERE locations.country_id = state1_countries.id AND locations.is_active = 1) as location_count')
                )
                ->where('state1_countries.is_active', 1)
                ->orderByDesc('location_count')
                ->orderBy(DB::raw("COALESCE(state1_countries.name, state1_countries.name_en, state1_countries.name_ar)"))
                ->toSql(),
            'countries' => $countries->toArray()
        ]);

        // افزودن پرچم و نام پیش‌فرض برای هر زبان
        foreach ($countries as $country) {
            $country->flag = $this->isoToFlagEmoji($country->iso_code);
            // نام پیش‌فرض برای نمایش
            $country->name = app()->getLocale() === 'en' ? ($country->name_en ?? $country->name_fa) : (app()->getLocale() === 'ar' ? ($country->name_ar ?? $country->name_fa) : $country->name_fa);
        }
        return $countries;
    }



    /**
     * تبدیل کد ISO دو حرفی به ایموجی پرچم (سازگار با همه نسخه‌های PHP)
     * @param string $isoCode
     * @return string
     */
    private function isoToFlagEmoji($isoCode)
    {
        if (!$isoCode || strlen($isoCode) != 2) return '';
        $isoCode = strtoupper($isoCode);
        $first = ord($isoCode[0]) - 65 + 0x1F1E6;
        $second = ord($isoCode[1]) - 65 + 0x1F1E6;
        return html_entity_decode('&#' . $first . ';', ENT_NOQUOTES, 'UTF-8')
             . html_entity_decode('&#' . $second . ';', ENT_NOQUOTES, 'UTF-8');
    }

    /**
     * Get provinces by country for dynamic filters
     *
     * @param  int  $countryId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProvincesByCountry(Request $request, $countryId)
    {
        try {
            $locale = app()->getLocale();
            $nameColumn = 'name';
            if ($locale !== 'fa') {
                $nameColumn = 'name_' . $locale;
            }

            $provinces = DB::table('state2_provinces')
                ->select(
                    'state2_provinces.id',
                    DB::raw("COALESCE(state2_provinces.{$nameColumn}, state2_provinces.name_en, state2_provinces.name) as name"),
                    'state2_provinces.code as emoji',
                    DB::raw('(SELECT COUNT(*) FROM locations WHERE locations.province_id = state2_provinces.id AND locations.is_active = 1) as location_count')
                )
                ->where('state2_provinces.country_id', $countryId)
                ->where('state2_provinces.is_active', 1)
                ->having('location_count', '>', 0) // Only show provinces with locations
                ->orderByDesc('location_count')
                ->orderBy(DB::raw("COALESCE(state2_provinces.{$nameColumn}, state2_provinces.name_en, state2_provinces.name)"))
                ->get();

            return response()->json([
                'success' => true,
                'data' => $provinces,
                'count' => $provinces->count()
            ]);
        } catch (\Exception $e) {
            \Log::error('Error getting provinces by country: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت استان‌ها',
                'data' => []
            ]);
        }
    }

    /**
     * Get cities by province for dynamic filters
     *
     * @param  int  $provinceId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCitiesByProvince(Request $request, $provinceId)
    {
        try {
            $locale = app()->getLocale();
            $nameColumn = 'name';
            if ($locale !== 'fa') {
                $nameColumn = 'name_' . $locale;
            }

            $cities = DB::table('state5_cities')
                ->select(
                    'state5_cities.id',
                    DB::raw("COALESCE(state5_cities.{$nameColumn}, state5_cities.name_en, state5_cities.name) as name"),
                    DB::raw('(SELECT COUNT(*) FROM locations WHERE locations.city_id = state5_cities.id AND locations.is_active = 1) as location_count')
                )
                ->where('state5_cities.province_id', $provinceId)
                ->where('state5_cities.is_active', 1)
                ->having('location_count', '>', 0) // Only show cities with locations
                ->orderByDesc('location_count')
                ->orderBy(DB::raw("COALESCE(state5_cities.{$nameColumn}, state5_cities.name_en, state5_cities.name)"))
                ->get();

            return response()->json([
                'success' => true,
                'data' => $cities,
                'count' => $cities->count()
            ]);
        } catch (\Exception $e) {
            \Log::error('Error getting cities by province: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت شهرها',
                'data' => []
            ]);
        }
    }

    /**
     * Get filter counts based on current filters
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getFilterCounts(Request $request)
    {
        try {
            $filters = $request->all();

            // Base query for active locations
            $baseQuery = Location::where('is_active', true);

            // Apply existing filters to get the filtered set
            if (!empty($filters['country'])) {
                $baseQuery->where('country_id', $filters['country']);
            }

            if (!empty($filters['province'])) {
                $baseQuery->where('province_id', $filters['province']);
            }

            if (!empty($filters['city'])) {
                $baseQuery->where('city_id', $filters['city']);
            }

            if (!empty($filters['rating'])) {
                $baseQuery->where('rating', '>=', $filters['rating']);
            }

            // Get location IDs that match current filters
            $filteredLocationIds = $baseQuery->pluck('id')->toArray();

            // Calculate counts for each category
            $categoryCounts = [];

            // Get all categories with their location counts
            $categories = Category::where('show_in_filters', true)
                ->withCount(['locations' => function ($query) use ($filteredLocationIds) {
                    $query->whereIn('id', $filteredLocationIds);
                }])
                ->get();

            foreach ($categories as $category) {
                $categoryCounts[$category->id] = $category->locations_count;
            }

            // Add total count
            $categoryCounts['all'] = count($filteredLocationIds);

            return response()->json([
                'success' => true,
                'counts' => $categoryCounts,
                'total' => count($filteredLocationIds),
                'applied_filters' => $filters
            ]);

        } catch (\Exception $e) {
            \Log::error('Error getting filter counts: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت آمار فیلترها',
                'counts' => []
            ]);
        }
    }

    /**
     * Get counties by province
     *
     * @param  int  $provinceId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCountiesByProvince(Request $request, $provinceId)
    {
        $locale = app()->getLocale();
        $nameColumn = 'name';
        if ($locale !== 'fa') {
            $nameColumn = 'name_' . $locale;
        }
        $counties = DB::table('state3_counties')
            ->select('state3_counties.id',
                DB::raw("COALESCE(state3_counties.{$nameColumn}, state3_counties.name_en, state3_counties.name) as name"),
                DB::raw('(SELECT COUNT(*) FROM locations WHERE locations.county_id = state3_counties.id AND locations.is_active = 1) as location_count'))
            ->where('state3_counties.province_id', $provinceId)
            ->where('state3_counties.is_active', 1)
            ->orderByDesc('location_count')
            ->orderBy(DB::raw("COALESCE(state3_counties.{$nameColumn}, state3_counties.name_en, state3_counties.name)"))
            ->get();
        return response()->json([
            'success' => true,
            'data' => $counties,
        ]);
    }

    /**
     * Get districts by county
     *
     * @param  int  $countyId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDistrictsByCounty(Request $request, $countyId)
    {
        $locale = app()->getLocale();
        $nameColumn = 'name';
        if ($locale !== 'fa') {
            $nameColumn = 'name_' . $locale;
        }
        $districts = DB::table('state4_districts')
            ->select('state4_districts.id',
                DB::raw("COALESCE(state4_districts.{$nameColumn}, state4_districts.name_en, state4_districts.name) as name"),
                DB::raw('(SELECT COUNT(*) FROM locations WHERE locations.district_id = state4_districts.id AND locations.is_active = 1) as location_count'))
            ->where('state4_districts.county_id', $countyId)
            ->where('state4_districts.is_active', 1)
            ->orderByDesc('location_count')
            ->orderBy(DB::raw("COALESCE(state4_districts.{$nameColumn}, state4_districts.name_en, state4_districts.name)"))
            ->get();
        return response()->json([
            'success' => true,
            'data' => $districts,
        ]);
    }

    /**
     * Get cities by district
     *
     * @param  int  $districtId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCitiesByDistrict(Request $request, $districtId)
    {
        $locale = app()->getLocale();
        $nameColumn = 'name';
        if ($locale !== 'fa') {
            $nameColumn = 'name_' . $locale;
        }
        $cities = DB::table('state5_cities')
            ->select('state5_cities.id',
                DB::raw("COALESCE(state5_cities.{$nameColumn}, state5_cities.name_en, state5_cities.name) as name"),
                DB::raw('(SELECT COUNT(*) FROM locations WHERE locations.city_id = state5_cities.id AND locations.is_active = 1) as location_count'))
            ->where('state5_cities.district_id', $districtId)
            ->where('state5_cities.is_active', 1)
            ->orderByDesc('location_count')
            ->orderBy(DB::raw("COALESCE(state5_cities.{$nameColumn}, state5_cities.name_en, state5_cities.name)"))
            ->get();
        return response()->json([
            'success' => true,
            'data' => $cities,
        ]);
    }

    /**
     * Get villages by district
     *
     * @param  int  $districtId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getVillagesByDistrict(Request $request, $districtId)
    {
        $locale = app()->getLocale();
        $nameColumn = 'name';
        if ($locale !== 'fa') {
            $nameColumn = 'name_' . $locale;
        }
        // اصلاح کوئری: استفاده از سه ستون در COALESCE
        $villages = DB::table('state6_village')
            ->select('state6_village.id',
                DB::raw("COALESCE(state6_village.{$nameColumn}, state6_village.name_en, state6_village.name) as name"),
                DB::raw('(SELECT COUNT(*) FROM locations WHERE locations.village_id = state6_village.id AND locations.is_active = 1) as location_count'))
            ->where('state6_village.district_id', $districtId)
            ->where('state6_village.is_active', 1)
            ->orderByDesc('location_count')
            ->orderBy(DB::raw("COALESCE(state6_village.{$nameColumn}, state6_village.name_en, state6_village.name)"))
            ->get();

        return response()->json([
            'success' => true,
            'data' => $villages,
        ]);
    }

    /**
     * Get zones by country
     *
     * @param  int  $countryId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getZonesByCountry(Request $request, $countryId)
    {
        $locale = app()->getLocale();
        $nameColumn = 'name';
        if ($locale !== 'fa') {
            $nameColumn = 'name_' . $locale;
        }
        $zones = DB::table('state7_zone')
            ->select('state7_zone.id',
                DB::raw("COALESCE(state7_zone.{$nameColumn}, state7_zone.name_en, state7_zone.name) as name"),
                DB::raw('(SELECT COUNT(*) FROM locations WHERE locations.zone_id = state7_zone.id AND locations.is_active = 1) as location_count'))
            ->where('state7_zone.country_id', $countryId)
            ->where('state7_zone.is_active', 1)
            ->orderByDesc('location_count')
            ->orderBy(DB::raw("COALESCE(state7_zone.{$nameColumn}, state7_zone.name_en, state7_zone.name)"))
            ->get();
        return response()->json([
            'success' => true,
            'data' => $zones,
        ]);
    }

    /**
     * API endpoint to get zones for a specific city
     *
     * @param  int  $cityId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getZonesByCity(Request $request, $cityId)
    {
        try {
            $zones = Zone::where('city_id', $cityId)
                ->orderBy('name_fa')
                ->get(['id', 'name_fa', 'name_en', 'zone_type']);

            return response()->json($zones);
        } catch (\Exception $e) {
            \Log::error('Error in getZonesByCity: '.$e->getMessage());

            return response()->json(['error' => 'Could not load zones'], 500);
        }
    }

    /**
     * Test method to diagnose location issues and create a test location if needed
     */
    public function testLocations()
    {
        $locationCount = Location::where('is_active', true)->count();

        $report = [
            'total_locations' => Location::count(),
            'active_locations' => $locationCount,
            'categories' => Category::count(),
            'active_categories' => Category::where('is_active', true)->count(),
            'countries' => Country::count(),
            'provinces' => Province::count(),
            'cities' => City::count(),
            'first_few_locations' => Location::take(3)->get(['id', 'title', 'is_active', 'lat', 'lng']),
        ];

        // If no active locations exist, create a test location
        if ($locationCount == 0) {
            try {
                // Find a valid category
                $category = Category::where('is_active', true)->first();
                if (! $category) {
                    $category = Category::first();
                    if ($category) {
                        $category->is_active = true;
                        $category->save();
                    }
                }

                // Find a country
                $country = Country::first();

                // Create test location
                $location = new Location;
                $location->title = 'تست لوکیشن';
                $location->description = 'این یک مکان تست است';
                $location->address = 'آدرس تست';
                $location->lat = 35.6892;  // Tehran coordinates
                $location->lng = 51.3890;
                $location->is_active = true;

                if ($category) {
                    $location->category_id = $category->id;
                }

                if ($country) {
                    $location->country_id = $country->id;
                }

                $location->save();

                $report['created_test_location'] = true;
                $report['test_location'] = $location;

                return response()->json($report);
            } catch (\Exception $e) {
                return response()->json([
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                    'report' => $report,
                ]);
            }
        }

        return response()->json($report);
    }

    /**
     * Get text advertisements for the ticker.
     *
     * @return \Illuminate\Http\Response
     */
    public function getTextAdvertisements()
    {
        try {
            $ads = \App\Models\Advertisement::active()
                ->where('position', 'text_ticker')
                ->where('is_active', true)
                ->select('id', 'title as content', 'target_url', 'speed', 'direction', 'animation_type', 'text_color')
                ->orderBy('display_order')
                ->limit(10)
                ->get();

            // Log what we found
            \Log::info('Text advertisements query', [
                'count' => $ads->count(),
                'ads' => $ads->toArray(),
            ]);

            // If no ads found, provide default fallback
            if ($ads->isEmpty()) {
                \Log::info('No text advertisements found, using fallback');
                $ads = [
                    [
                        'id' => 1,
                        'content' => 'به اَلو ایران خوش آمدید! مرجع جامع معرفی مکان‌های گردشگری و تفریحی ایران',
                        'target_url' => null,
                        'speed' => 20,
                        'direction' => 'rtl',
                        'animation_type' => 'none',
                        'text_color' => '#FFC107',
                    ],
                ];
            }

            return response()->json(['success' => true, 'ads' => $ads]);
        } catch (\Exception $e) {
            \Log::error('Error in getTextAdvertisements: '.$e->getMessage());

            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'ads' => [
                    [
                        'id' => 1,
                        'content' => 'به اَلو ایران خوش آمدید! مرجع جامع معرفی مکان‌های گردشگری و تفریحی ایران',
                        'target_url' => null,
                        'speed' => 20,
                        'direction' => 'rtl',
                        'animation_type' => 'none',
                        'text_color' => '#FFC107',
                    ],
                ],
            ]);
        }
    }

    /**
     * Display the categories page.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function categories()
    {
        // Get all parent categories with their children
        $categories = Category::where('parent_id', null)
            ->with(['children' => function ($query) {
                $query->orderBy('sort_order');
            }])
            ->orderBy('sort_order')
            ->get();

        return view('templates.categories', compact('categories'));
    }

    /**
     * Display the map page.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function map()
    {
        $locations = Location::with(['categoryRelation.activeTemplateRelation', 'city', 'province'])
            ->where('is_active', true)
            ->get();

        return view('templates.map', compact('locations'));
    }

    /**
     * Display the popular places page with advanced filtering and statistics.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function popular(Request $request)
    {
        // Get filter parameters
        $categoryId = $request->get('category');
        $provinceId = $request->get('province');
        $cityId = $request->get('city');
        $sortBy = $request->get('sort', 'star_rating');
        $perPage = $request->get('per_page', 12);
        $minRating = $request->get('min_rating', 0);

        // Build query with filters
        $query = Location::with([
            'categoryRelation.activeTemplateRelation',
            'city',
            'province',
            'ratings' => function($q) {
                $q->where('status', 'approved')->select('title_id', 'rating', 'overall_rating');
            }
        ])
        ->where('is_active', true)
        ->where('star_rating', '>=', $minRating);

        // Apply filters
        if ($categoryId) {
            $query->where('category_id', $categoryId);
        }

        if ($provinceId) {
            $query->where('province_id', $provinceId);
        }

        if ($cityId) {
            $query->where('city_id', $cityId);
        }

        // Apply sorting
        switch ($sortBy) {
            case 'newest':
                $query->orderBy('created_at', 'desc');
                break;
            case 'oldest':
                $query->orderBy('created_at', 'asc');
                break;
            case 'name_asc':
                $query->orderBy('title', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('title', 'desc');
                break;
            case 'view_count':
                // Since view_count field doesn't exist, we'll use created_at as fallback
                $query->orderBy('created_at', 'desc');
                break;
            default: // star_rating
                $query->orderBy('star_rating', 'desc')
                      ->orderBy('created_at', 'desc');
        }

        // Get paginated results
        $popularLocations = $query->paginate($perPage);

        // Calculate average rating for each location
        foreach ($popularLocations as $location) {
            if ($location->ratings->count() > 0) {
                $location->average_rating = $location->ratings->avg('rating');
                $location->ratings_count = $location->ratings->count();
            } else {
                $location->average_rating = $location->star_rating ?? 0;
                $location->ratings_count = 0;
            }
        }

        // Get statistics
        $totalLocations = Location::where('is_active', true)->count();
        $totalCategories = \DB::table('locations')
            ->where('is_active', true)
            ->distinct('category_id')
            ->count('category_id');
        $totalProvinces = \DB::table('locations')
            ->where('is_active', true)
            ->distinct('province_id')
            ->count('province_id');
        $averageRating = Location::where('is_active', true)
            ->where('star_rating', '>', 0)
            ->avg('star_rating');

        // Get filter options
        $categories = \App\Models\Category::where('is_active', true)
            ->orderBy('category_fa')
            ->get();

        $provinces = \App\Models\Province::where('is_active', true)
            ->orderBy('name')
            ->get();

        $cities = collect();
        if ($provinceId) {
            $cities = \App\Models\City::where('is_active', true)
                ->where('province_id', $provinceId)
                ->orderBy('name')
                ->get();
        }

        return view('templates.popular', compact(
            'popularLocations',
            'categories',
            'provinces',
            'cities',
            'totalLocations',
            'totalCategories',
            'totalProvinces',
            'averageRating',
            'categoryId',
            'provinceId',
            'cityId',
            'sortBy',
            'minRating'
        ));
    }

    /**
     * Display the about us page.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function about()
    {
        // Get comprehensive statistics for about page
        $stats = [
            'total_locations' => Location::where('is_active', true)->count(),
            'total_categories' => \App\Models\Category::where('is_active', true)->count(),
            'total_cities' => \App\Models\City::where('is_active', true)->count(),
            'total_provinces' => \App\Models\Province::where('is_active', true)->count(),
            'total_ratings' => \App\Models\Rating::where('status', 'approved')->count(),
            'total_users' => \App\Models\User::where('status', 'active')->count(),
            'average_rating' => round(\App\Models\Rating::where('status', 'approved')->avg('rating') ?? 0, 1),
            'diamond_locations' => Location::where('package_type', 'diamond')->where('is_active', true)->count(),
            'gold_locations' => Location::where('package_type', 'gold')->where('is_active', true)->count(),
            'recent_locations' => Location::where('is_active', true)->where('created_at', '>=', now()->subDays(30))->count(),
            'total_visits' => $this->getTotalVisits(),
            'monthly_growth' => $this->getMonthlyGrowth(),
        ];

        // Get popular categories with location counts
        $popularCategories = \App\Models\Category::withCount(['locations' => function($query) {
            $query->where('is_active', true);
        }])
        ->where('is_active', true)
        ->orderBy('locations_count', 'desc')
        ->take(6)
        ->get();

        // Get top provinces by location count
        $topProvinces = \App\Models\Province::withCount(['locations' => function($query) {
            $query->where('is_active', true);
        }])
        ->where('is_active', true)
        ->orderBy('locations_count', 'desc')
        ->take(5)
        ->get();

        // Get recent achievements/milestones
        $achievements = [
            [
                'title' => 'راه‌اندازی سامانه',
                'description' => 'شروع فعالیت اَلو ایران با هدف معرفی مکان‌های گردشگری',
                'date' => '1402/01/01',
                'icon' => 'fas fa-rocket'
            ],
            [
                'title' => 'اولین هزار مکان',
                'description' => 'ثبت اولین هزار مکان گردشگری در سامانه',
                'date' => '1402/06/15',
                'icon' => 'fas fa-map-marker-alt'
            ],
            [
                'title' => 'پوشش سراسری',
                'description' => 'گسترش خدمات به تمامی استان‌های کشور',
                'date' => '1402/09/20',
                'icon' => 'fas fa-globe'
            ],
            [
                'title' => 'اپلیکیشن موبایل',
                'description' => 'عرضه اپلیکیشن موبایل برای دسترسی آسان‌تر',
                'date' => '1403/02/10',
                'icon' => 'fas fa-mobile-alt'
            ]
        ];

        return view('templates.about', compact('stats', 'popularCategories', 'topProvinces', 'achievements'));
    }

    /**
     * Get total visits count (with fallback)
     */
    private function getTotalVisits()
    {
        try {
            return \App\Models\Visit::count();
        } catch (\Exception $e) {
            // Fallback calculation based on locations
            return Location::where('is_active', true)->count() * 25;
        }
    }

    /**
     * Calculate monthly growth percentage
     */
    private function getMonthlyGrowth()
    {
        try {
            $currentMonth = Location::where('is_active', true)
                ->where('created_at', '>=', now()->startOfMonth())
                ->count();

            $lastMonth = Location::where('is_active', true)
                ->whereBetween('created_at', [
                    now()->subMonth()->startOfMonth(),
                    now()->subMonth()->endOfMonth()
                ])
                ->count();

            if ($lastMonth == 0) return 0;

            return round((($currentMonth - $lastMonth) / $lastMonth) * 100, 1);
        } catch (\Exception $e) {
            return rand(5, 15); // Fallback growth rate
        }
    }

    /**
     * Display the contact us page.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function contact()
    {
        return view('templates.contact');
    }

    /**
     * Handle contact form submission.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function contactSubmit(Request $request)
    {
        // Validate the form data
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
        ]);

        // Here you would typically save the contact form data to the database
        // or send an email to the site admin

        // For demonstration, just return with a success message
        return redirect()->route('contact')->with('success', 'پیام شما با موفقیت ارسال شد. با تشکر از تماس شما.');
    }
}
