<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class BackupController extends Controller
{
    /**
     * نمایش صفحه مدیریت پشتیبان‌گیری
     */
    public function index()
    {
        $backups = $this->getBackups();
        return view('admin.backup.index', compact('backups'));
    }

    /**
     * ایجاد پشتیبان جدید
     */
    public function create()
    {
        try {
            Artisan::call('backup:run');
            return redirect()->route('admin.backup.index')
                ->with('success', 'پشتیبان با موفقیت ایجاد شد.');
        } catch (\Exception $e) {
            return redirect()->route('admin.backup.index')
                ->with('error', 'خطا در ایجاد پشتیبان: ' . $e->getMessage());
        }
    }

    /**
     * دانلود پشتیبان
     */
    public function download($filename)
    {
        $path = storage_path('app/Laravel/' . $filename);
        
        if (!file_exists($path)) {
            return redirect()->route('admin.backup.index')
                ->with('error', 'فایل پشتیبان یافت نشد.');
        }

        return response()->download($path);
    }

    /**
     * حذف پشتیبان
     */
    public function destroy($filename)
    {
        try {
            $path = storage_path('app/Laravel/' . $filename);
            
            if (file_exists($path)) {
                unlink($path);
                return redirect()->route('admin.backup.index')
                    ->with('success', 'پشتیبان با موفقیت حذف شد.');
            }

            return redirect()->route('admin.backup.index')
                ->with('error', 'فایل پشتیبان یافت نشد.');
        } catch (\Exception $e) {
            return redirect()->route('admin.backup.index')
                ->with('error', 'خطا در حذف پشتیبان: ' . $e->getMessage());
        }
    }

    /**
     * دریافت لیست پشتیبان‌ها
     */
    private function getBackups()
    {
        $backups = [];
        $path = storage_path('app/Laravel');

        if (is_dir($path)) {
            $files = scandir($path);
            foreach ($files as $file) {
                if ($file != '.' && $file != '..') {
                    $backups[] = [
                        'filename' => $file,
                        'size' => $this->formatSize(filesize($path . '/' . $file)),
                        'created_at' => Carbon::createFromTimestamp(filemtime($path . '/' . $file))->format('Y-m-d H:i:s')
                    ];
                }
            }
        }

        return $backups;
    }

    /**
     * تبدیل حجم فایل به فرمت خوانا
     */
    private function formatSize($size)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $i = 0;
        while ($size >= 1024 && $i < count($units) - 1) {
            $size /= 1024;
            $i++;
        }
        return round($size, 2) . ' ' . $units[$i];
    }
} 