<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class SiteSettings
{
    /**
     * دریافت تنظیمات فعال سایت برای زبان فعلی
     * 
     * @param string $language_code کد زبان (اختیاری)
     * @return array تنظیمات فعال سایت
     */
    public static function getActiveSettings($language_code = null)
    {
        // اگر کد زبان مشخص نشده باشد، از زبان پیش‌فرض استفاده می‌شود
        if ($language_code === null) {
            $language_code = app()->getLocale();
        }
        
        // دریافت تنظیمات از کش
        $cache_key = 'site_settings_' . $language_code;
        $cached_settings = Cache::get($cache_key);
        
        if ($cached_settings !== null) {
            return $cached_settings;
        }
        
        // دریافت تنظیمات از دیتابیس
        $settings = DB::table('site_settings')
            ->where('language_code', $language_code)
            ->where('is_active', 1)
            ->pluck('setting_value', 'setting_key')
            ->toArray();
        
        // تبدیل تنظیمات JSON به آرایه
        $json_fields = [
            'social_media',
            'smtp_settings',
            'payment_settings',
            'notification_settings',
            'cache_settings',
            'security_settings',
            'api_settings',
            'seo_settings',
            'display_settings',
            'notification_templates',
            'custom_fields',
            'integration_settings',
            'backup_settings',
            'monitoring_settings',
            'reporting_settings',
            'localization_settings',
            'performance_settings',
            'accessibility_settings',
            'compliance_settings',
            'advanced_settings'
        ];
        
        foreach ($json_fields as $field) {
            if (isset($settings[$field])) {
                $settings[$field] = json_decode($settings[$field], true) ?? [];
            }
        }
        
        // ذخیره در کش
        Cache::put($cache_key, $settings, 3600); // کش برای یک ساعت
        
        return $settings;
    }
    
    /**
     * دریافت مقدار یک تنظیم خاص
     * 
     * @param string $key کلید تنظیم
     * @param mixed $default مقدار پیش‌فرض
     * @param string $language_code کد زبان (اختیاری)
     * @return mixed مقدار تنظیم
     */
    public static function getSetting($key, $default = null, $language_code = null)
    {
        $settings = self::getActiveSettings($language_code);
        return $settings[$key] ?? $default;
    }
} 