@extends('admin.layouts.app')

@section('title', 'گزارش مکان‌ها - پنل مدیریت')

@push('styles')
<link rel="stylesheet" href="{{ asset('css/fonts.css') }}">
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap5.min.css">
<link rel="stylesheet" href="https://cdn.datatables.net/responsive/2.5.0/css/responsive.bootstrap5.min.css">
<link rel="stylesheet" href="https://cdn.datatables.net/buttons/2.4.2/css/buttons.bootstrap5.min.css">
<style>
:root {
    --primary-color: #4e73df;
    --success-color: #1cc88a;
    --info-color: #36b9cc;
    --warning-color: #f6c23e;
    --danger-color: #e74a3b;
    --light-color: #f8f9fc;
    --dark-color: #5a5c69;
    --border-color: #e3e6f0;
    --shadow-color: rgba(0, 0, 0, 0.15);
}

body {
    font-family: 'IRANSans', Tahoma, Arial, sans-serif;
    background-color: var(--light-color);
}

.modern-card {
    background: white;
    border-radius: 12px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    border: none;
    transition: all 0.3s ease;
    overflow: hidden;
}

.modern-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
}

.stats-card {
    background: linear-gradient(135deg, var(--primary-color) 0%, #224abe 100%);
    color: white;
    border-radius: 16px;
    padding: 2rem;
    position: relative;
    overflow: hidden;
}

.stats-card::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 100%;
    height: 100%;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 50%;
    transform: rotate(45deg);
}

.stats-number {
    font-size: 2.5rem;
    font-weight: bold;
    margin-bottom: 0.5rem;
}

.stats-label {
    font-size: 1rem;
    opacity: 0.9;
}

.filter-section {
    background: white;
    border-radius: 12px;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
}

.filter-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--dark-color);
    margin-bottom: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.modern-select, .modern-input {
    border: 2px solid var(--border-color);
    border-radius: 8px;
    padding: 0.75rem 1rem;
    font-size: 0.95rem;
    transition: all 0.3s ease;
    background: white;
}

.modern-select:focus, .modern-input:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(78, 115, 223, 0.1);
    outline: none;
}

.modern-btn {
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    font-weight: 500;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
}

.btn-primary-modern {
    background: linear-gradient(135deg, var(--primary-color) 0%, #224abe 100%);
    color: white;
}

.btn-primary-modern:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 15px rgba(78, 115, 223, 0.3);
}

.chart-container {
    background: white;
    border-radius: 12px;
    padding: 2rem;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
    margin-bottom: 2rem;
}

.chart-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--dark-color);
    margin-bottom: 1.5rem;
    text-align: center;
}

.table-container {
    background: white;
    border-radius: 12px;
    padding: 2rem;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
}

.modern-table {
    border-collapse: separate;
    border-spacing: 0;
}

.modern-table thead th {
    background: var(--light-color);
    color: var(--dark-color);
    font-weight: 600;
    padding: 1rem;
    border: none;
    position: relative;
}

.modern-table tbody td {
    padding: 1rem;
    border-bottom: 1px solid var(--border-color);
    vertical-align: middle;
}

.modern-table tbody tr:hover {
    background-color: rgba(78, 115, 223, 0.05);
}

.status-badge {
    padding: 0.4rem 0.8rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 500;
    text-align: center;
    min-width: 80px;
    display: inline-block;
}

.badge-active {
    background: rgba(28, 200, 138, 0.1);
    color: var(--success-color);
}

.badge-inactive {
    background: rgba(231, 74, 59, 0.1);
    color: var(--danger-color);
}

.package-badge {
    padding: 0.3rem 0.7rem;
    border-radius: 15px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
}

.package-diamond {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.package-gold {
    background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
    color: white;
}

.package-silver {
    background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
    color: white;
}

.package-bronze {
    background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);
    color: white;
}

.package-regular {
    background: #f8f9fa;
    color: #6c757d;
    border: 1px solid #dee2e6;
}

.loading-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(255, 255, 255, 0.9);
    display: flex;
    justify-content: center;
    align-items: center;
    z-index: 9999;
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
}

.loading-overlay.show {
    opacity: 1;
    visibility: visible;
}

.spinner {
    width: 50px;
    height: 50px;
    border: 4px solid var(--border-color);
    border-top: 4px solid var(--primary-color);
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.fade-in {
    animation: fadeIn 0.5s ease-in;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

@media (max-width: 768px) {
    .stats-card {
        margin-bottom: 1rem;
    }

    .filter-section {
        padding: 1rem;
    }

    .chart-container, .table-container {
        padding: 1rem;
    }
}
</style>
@endpush

@section('content')
<div class="loading-overlay" id="loadingOverlay">
    <div class="spinner"></div>
</div>

<div class="container-fluid px-4">
    <!-- Header Section -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="fas fa-chart-bar text-primary me-2"></i>
                گزارش جامع مکان‌ها
            </h1>
            <p class="text-muted mt-1">تحلیل و بررسی اطلاعات مکان‌های ثبت شده</p>
        </div>
        <div class="d-flex gap-2">
            <button class="modern-btn btn-primary-modern" onclick="window.print()">
                <i class="fas fa-print me-1"></i>
                چاپ گزارش
            </button>
            <button class="modern-btn btn-primary-modern" onclick="refreshTable()">
                <i class="fas fa-sync-alt me-1"></i>
                بروزرسانی
            </button>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="stats-card fade-in">
                <div class="stats-number" id="totalLocations">{{ number_format($stats['total']) }}</div>
                <div class="stats-label">
                    <i class="fas fa-map-marker-alt me-1"></i>
                    کل مکان‌ها
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="stats-card fade-in" style="background: linear-gradient(135deg, var(--success-color) 0%, #17a673 100%);">
                <div class="stats-number" id="activeLocations">{{ number_format($stats['active']) }}</div>
                <div class="stats-label">
                    <i class="fas fa-check-circle me-1"></i>
                    مکان‌های فعال
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="stats-card fade-in" style="background: linear-gradient(135deg, var(--warning-color) 0%, #d4a017 100%);">
                <div class="stats-number" id="premiumLocations">{{ number_format($stats['premium']) }}</div>
                <div class="stats-label">
                    <i class="fas fa-crown me-1"></i>
                    مکان‌های ویژه
                </div>
            </div>
        </div>
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="stats-card fade-in" style="background: linear-gradient(135deg, var(--info-color) 0%, #2c9faf 100%);">
                <div class="stats-number" id="todayLocations">{{ number_format($stats['today']) }}</div>
                <div class="stats-label">
                    <i class="fas fa-calendar-day me-1"></i>
                    امروز ثبت شده
                </div>
            </div>
        </div>
    </div>

    <!-- Advanced Filters -->
    <div class="filter-section fade-in">
        <div class="filter-title">
            <i class="fas fa-filter"></i>
            فیلترهای پیشرفته
        </div>
        <form id="filterForm" action="{{ route('admin.reports.locations') }}" method="GET">
            <div class="row">
                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">دسته‌بندی</label>
                    <select name="category_id" id="category_id" class="form-select modern-select">
                        <option value="">همه دسته‌ها</option>
                        @foreach($categories as $category)
                            <option value="{{ $category->id }}" {{ request('category_id') == $category->id ? 'selected' : '' }}>
                                {{ $category->getTranslatedName() ?? $category->category_fa ?? 'نامشخص' }}
                            </option>
                        @endforeach
                    </select>
                </div>

                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">کشور</label>
                    <select name="country_id" id="country_id" class="form-select modern-select">
                        <option value="">همه کشورها</option>
                        @foreach($countries as $country)
                            <option value="{{ $country->id }}" {{ request('country_id') == $country->id ? 'selected' : '' }}>
                                {{ $country->getTranslatedName() ?? $country->name ?? 'نامشخص' }}
                            </option>
                        @endforeach
                    </select>
                </div>

                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">استان</label>
                    <select name="province_id" id="province_id" class="form-select modern-select">
                        <option value="">همه استان‌ها</option>
                        @foreach($provinces as $province)
                            <option value="{{ $province->id }}" {{ request('province_id') == $province->id ? 'selected' : '' }}>
                                {{ $province->getTranslatedName() ?? $province->name ?? 'نامشخص' }}
                            </option>
                        @endforeach
                    </select>
                </div>

                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">شهر</label>
                    <select name="city_id" id="city_id" class="form-select modern-select">
                        <option value="">همه شهرها</option>
                        @foreach($cities as $city)
                            <option value="{{ $city->id }}" {{ request('city_id') == $city->id ? 'selected' : '' }}>
                                {{ $city->getTranslatedName() ?? $city->name ?? 'نامشخص' }}
                            </option>
                        @endforeach
                    </select>
                </div>

                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">روستا</label>
                    <select name="village_id" id="village_id" class="form-select modern-select">
                        <option value="">همه روستاها</option>
                        @foreach($villages as $village)
                            <option value="{{ $village->id }}" {{ request('village_id') == $village->id ? 'selected' : '' }}>
                                {{ $village->getTranslatedName() ?? $village->name ?? 'نامشخص' }}
                            </option>
                        @endforeach
                    </select>
                </div>

                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">وضعیت</label>
                    <select name="status" id="status" class="form-select modern-select">
                        <option value="">همه وضعیت‌ها</option>
                        <option value="1" {{ request('status') === '1' ? 'selected' : '' }}>فعال</option>
                        <option value="0" {{ request('status') === '0' ? 'selected' : '' }}>غیرفعال</option>
                    </select>
                </div>

                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">نوع پکیج</label>
                    <select name="package_type" id="package_type" class="form-select modern-select">
                        <option value="">همه پکیج‌ها</option>
                        <option value="diamond" {{ request('package_type') == 'diamond' ? 'selected' : '' }}>💎 الماس</option>
                        <option value="gold" {{ request('package_type') == 'gold' ? 'selected' : '' }}>🥇 طلایی</option>
                        <option value="silver" {{ request('package_type') == 'silver' ? 'selected' : '' }}>🥈 نقره‌ای</option>
                        <option value="bronze" {{ request('package_type') == 'bronze' ? 'selected' : '' }}>🥉 برنزی</option>
                        <option value="regular" {{ request('package_type') == 'regular' ? 'selected' : '' }}>📋 عادی</option>
                    </select>
                </div>

                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">از تاریخ</label>
                    <input type="date" name="date_from" id="date_from" class="form-control modern-input admin-datepicker" value="{{ request('date_from') }}" placeholder="انتخاب تاریخ">
                </div>

                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">تا تاریخ</label>
                    <input type="date" name="date_to" id="date_to" class="form-control modern-input admin-datepicker" value="{{ request('date_to') }}" placeholder="انتخاب تاریخ">
                </div>

                <div class="col-md-3 mb-3">
                    <label class="form-label fw-semibold">جستجو در نام</label>
                    <input type="text" name="search" id="search" class="form-control modern-input" value="{{ request('search') }}" placeholder="نام مکان را وارد کنید...">
                </div>

                <div class="col-md-3 mb-3 d-flex align-items-end">
                    <div class="d-flex gap-2 w-100">
                        <button type="submit" class="modern-btn btn-primary-modern flex-fill">
                            <i class="fas fa-search me-1"></i>
                            اعمال فیلتر
                        </button>
                        <a href="{{ route('admin.reports.locations') }}" class="modern-btn" style="background: #6c757d; color: white;">
                            <i class="fas fa-refresh"></i>
                        </a>
                    </div>
                </div>
            </div>
        </form>
    </div>
    <!-- Charts Section -->
    <div class="row mb-4">
        <div class="col-lg-6 mb-4">
            <div class="chart-container fade-in">
                <div class="chart-title">
                    <i class="fas fa-chart-pie text-primary me-2"></i>
                    توزیع مکان‌ها بر اساس دسته‌بندی
                </div>
                <div style="position: relative; height: 300px;">
                    <canvas id="categoryChart"></canvas>
                </div>
            </div>
        </div>
        <div class="col-lg-6 mb-4">
            <div class="chart-container fade-in">
                <div class="chart-title">
                    <i class="fas fa-chart-bar text-success me-2"></i>
                    توزیع مکان‌ها بر اساس استان
                </div>
                <div style="position: relative; height: 300px;">
                    <canvas id="provinceChart"></canvas>
                </div>
            </div>
        </div>
    </div>

    <!-- Country and Package Distribution Charts -->
    <div class="row mb-4">
        <div class="col-lg-6 mb-4">
            <div class="chart-container fade-in">
                <div class="chart-title">
                    <i class="fas fa-globe text-info me-2"></i>
                    توزیع مکان‌ها بر اساس کشور
                </div>
                <div style="position: relative; height: 300px;">
                    <canvas id="countryChart"></canvas>
                </div>
            </div>
        </div>
        <div class="col-lg-6 mb-4">
            <div class="chart-container fade-in">
                <div class="chart-title">
                    <i class="fas fa-crown text-warning me-2"></i>
                    توزیع انواع پکیج‌ها
                </div>
                <div style="position: relative; height: 300px;">
                    <canvas id="packageChart"></canvas>
                </div>
            </div>
        </div>
    </div>

    <!-- Monthly Trend Chart -->
    <div class="row mb-4">
        <div class="col-12 mb-4">
            <div class="chart-container fade-in">
                <div class="chart-title">
                    <i class="fas fa-calendar-alt text-info me-2"></i>
                    روند ثبت مکان‌ها در ماه‌های اخیر
                </div>
                <div style="position: relative; height: 400px;">
                    <canvas id="monthlyChart"></canvas>
                </div>
            </div>
        </div>
    </div>

    <!-- Advanced Data Table -->
    <div class="table-container fade-in">
        <div class="d-flex justify-content-between align-items-center mb-3">
            <h5 class="mb-0">
                <i class="fas fa-table text-primary me-2"></i>
                جدول تفصیلی مکان‌ها
            </h5>
            <div class="d-flex gap-2">
                <button class="modern-btn" style="background: var(--success-color); color: white;" onclick="exportTableData()">
                    <i class="fas fa-download me-1"></i>
                    دانلود داده‌ها
                </button>
                <button class="modern-btn" style="background: var(--info-color); color: white;" onclick="refreshTable()">
                    <i class="fas fa-sync-alt me-1"></i>
                    بروزرسانی
                </button>
            </div>
        </div>

        <div class="table-responsive">
            <table id="locationsTable" class="table modern-table w-100">
                <thead>
                    <tr>
                        <th>ردیف</th>
                        <th>نام مکان</th>
                        <th>دسته‌بندی</th>
                        <th>کشور</th>
                        <th>استان</th>
                        <th>شهر</th>
                        <th>روستا</th>
                        <th>نوع پکیج</th>
                        <th>وضعیت</th>
                        <th>امتیاز</th>
                        <th>تاریخ ثبت</th>
                        <th>عملیات</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($locations as $index => $location)
                    <tr>
                        <td>{{ $locations->firstItem() + $index }}</td>
                        <td>
                            <div class="d-flex align-items-center">
                                @php
                                    $images = null;
                                    if ($location->images) {
                                        try {
                                            $images = is_string($location->images) ? json_decode($location->images, true) : $location->images;
                                        } catch (Exception $e) {
                                            $images = null;
                                        }
                                    }
                                @endphp

                                @if($images && is_array($images) && count($images) > 0)
                                    <img src="{{ asset('storage/' . $images[0]) }}"
                                         alt="{{ $location->title }}"
                                         class="rounded me-2"
                                         style="width: 40px; height: 40px; object-fit: cover;"
                                         onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                    <div class="bg-light rounded me-2 d-flex align-items-center justify-content-center"
                                         style="width: 40px; height: 40px; display: none;">
                                        <i class="fas fa-image text-muted"></i>
                                    </div>
                                @else
                                    <div class="bg-light rounded me-2 d-flex align-items-center justify-content-center"
                                         style="width: 40px; height: 40px;">
                                        <i class="fas fa-image text-muted"></i>
                                    </div>
                                @endif
                                <div>
                                    <div class="fw-semibold">{{ $location->title ?? 'بدون عنوان' }}</div>
                                    @if($location->Slogan)
                                        <small class="text-muted">{{ Str::limit($location->Slogan, 30) }}</small>
                                    @endif
                                </div>
                            </div>
                        </td>
                        <td>
                            @if($location->categoryRelation)
                                <span class="badge" style="background: {{ $location->categoryRelation->color ?? '#6c757d' }}; color: white;">
                                    {{ $location->categoryRelation->getTranslatedName() }}
                                </span>
                            @else
                                <span class="badge" style="background: #6c757d; color: white;">
                                    بدون دسته‌بندی
                                </span>
                            @endif
                        </td>
                        <td>
                            @if($location->country)
                                {{ $location->country->getTranslatedName() }}
                            @else
                                <span class="text-muted">نامشخص</span>
                            @endif
                        </td>
                        <td>
                            @if($location->province)
                                {{ $location->province->getTranslatedName() }}
                            @else
                                <span class="text-muted">نامشخص</span>
                            @endif
                        </td>
                        <td>
                            @if($location->city)
                                {{ $location->city->getTranslatedName() }}
                            @else
                                <span class="text-muted">نامشخص</span>
                            @endif
                        </td>
                        <td>
                            @if($location->village)
                                {{ $location->village->getTranslatedName() }}
                            @else
                                <span class="text-muted">نامشخص</span>
                            @endif
                        </td>
                        <td>
                            @switch($location->package_type)
                                @case('diamond')
                                    <span class="package-badge package-diamond">💎 الماس</span>
                                    @break
                                @case('gold')
                                    <span class="package-badge package-gold">🥇 طلایی</span>
                                    @break
                                @case('silver')
                                    <span class="package-badge package-silver">🥈 نقره‌ای</span>
                                    @break
                                @case('bronze')
                                    <span class="package-badge package-bronze">🥉 برنزی</span>
                                    @break
                                @default
                                    <span class="package-badge package-regular">📋 عادی</span>
                            @endswitch
                        </td>
                        <td>
                            @if($location->is_active)
                                <span class="status-badge badge-active">
                                    <i class="fas fa-check-circle me-1"></i>
                                    فعال
                                </span>
                            @else
                                <span class="status-badge badge-inactive">
                                    <i class="fas fa-times-circle me-1"></i>
                                    غیرفعال
                                </span>
                            @endif
                        </td>
                        <td>
                            @if(isset($location->star_rating) && $location->star_rating > 0)
                                <div class="d-flex align-items-center">
                                    @for($i = 1; $i <= 5; $i++)
                                        @if($i <= $location->star_rating)
                                            <i class="fas fa-star text-warning"></i>
                                        @else
                                            <i class="far fa-star text-muted"></i>
                                        @endif
                                    @endfor
                                    <span class="ms-1 small text-muted">({{ number_format($location->star_rating, 1) }})</span>
                                </div>
                            @else
                                <span class="text-muted">بدون امتیاز</span>
                            @endif
                        </td>
                        <td>
                            <span class="text-muted" data-date="{{ $location->created_at }}">
                                @if(app()->getLocale() == 'fa')
                                    {{ verta($location->created_at)->format('Y/m/d') }}
                                @else
                                    {{ $location->created_at->format('Y/m/d') }}
                                @endif
                            </span>
                        </td>
                        <td>
                            <div class="d-flex gap-1">
                                @if(Route::has('admin.locations.show'))
                                    <a href="{{ route('admin.locations.show', $location->id) }}"
                                       class="btn btn-sm btn-outline-primary"
                                       title="مشاهده">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                @endif
                                @if(Route::has('admin.locations.edit'))
                                    <a href="{{ route('admin.locations.edit', $location->id) }}"
                                       class="btn btn-sm btn-outline-warning"
                                       title="ویرایش">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                @endif
                                @if($location->lat && $location->lng)
                                    <button class="btn btn-sm btn-outline-info"
                                            onclick="showLocationOnMap({{ $location->lat }}, {{ $location->lng }})"
                                            title="نمایش روی نقشه">
                                        <i class="fas fa-map-marker-alt"></i>
                                    </button>
                                @else
                                    <button class="btn btn-sm btn-outline-secondary"
                                            disabled
                                            title="مختصات موجود نیست">
                                        <i class="fas fa-map-marker-alt"></i>
                                    </button>
                                @endif
                            </div>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <div class="d-flex justify-content-between align-items-center mt-4">
            <div class="text-muted">
                نمایش {{ $locations->firstItem() }} تا {{ $locations->lastItem() }} از {{ number_format($locations->total()) }} مکان
            </div>
            <div>
                {{ $locations->appends(request()->query())->links() }}
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/dataTables.responsive.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/responsive.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.colVis.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.53/pdfmake.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.53/vfs_fonts.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.html5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.print.min.js"></script>
<script src="{{ asset('js/admin-reports-locations.js') }}"></script>

<script>
// Global variables
let categoryChart, provinceChart, countryChart, packageChart, monthlyChart;
let locationsTable;

document.addEventListener('DOMContentLoaded', function() {
    // Initialize loading overlay
    showLoading();

    // Initialize charts
    initializeCharts();

    // Initialize DataTable
    initializeDataTable();

    // Initialize form interactions
    initializeFormInteractions();

    // Initialize auto-refresh
    initializeAutoRefresh();

    // Initialize number animations
    initializeNumberAnimations();

    // Hide loading overlay
    hideLoading();
});

// Loading functions
function showLoading() {
    const overlay = document.getElementById('loadingOverlay');
    if (overlay) overlay.classList.add('show');
}

function hideLoading() {
    const overlay = document.getElementById('loadingOverlay');
    if (overlay) overlay.classList.remove('show');
}

// Chart initialization
function initializeCharts() {
    const chartConfig = getChartConfig();

    // Category Chart
    const categoryCtx = document.getElementById('categoryChart');
    if (categoryCtx) {
        categoryChart = new Chart(categoryCtx.getContext('2d'), {
            type: 'doughnut',
            data: {
                labels: {!! json_encode($categoryStats->pluck('name')) !!},
                datasets: [{
                    data: {!! json_encode($categoryStats->pluck('count')) !!},
                    backgroundColor: [
                        '#4e73df', '#1cc88a', '#36b9cc', '#f6c23e', '#e74a3b',
                        '#858796', '#5a5c69', '#6f42c1', '#e83e8c', '#fd7e14'
                    ],
                    borderWidth: 0,
                    hoverBorderWidth: 3,
                    hoverBorderColor: '#fff'
                }]
            },
            options: {
                ...chartConfig,
                plugins: {
                    ...chartConfig.plugins,
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true,
                            font: {
                                family: 'IRANSans, sans-serif'
                            }
                        }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = ((context.parsed * 100) / total).toFixed(1);
                                return `${context.label}: ${formatNumber(context.parsed)} (${percentage}%)`;
                            }
                        }
                    }
                },
                cutout: '60%'
            }
        });
    }

    // Province Chart
    const provinceCtx = document.getElementById('provinceChart');
    if (provinceCtx) {
        provinceChart = new Chart(provinceCtx.getContext('2d'), {
            type: 'bar',
            data: {
                labels: {!! json_encode($provinceStats->pluck('name')) !!},
                datasets: [{
                    label: 'تعداد مکان‌ها',
                    data: {!! json_encode($provinceStats->pluck('count')) !!},
                    backgroundColor: 'rgba(78, 115, 223, 0.8)',
                    borderColor: 'rgba(78, 115, 223, 1)',
                    borderWidth: 1,
                    borderRadius: 4,
                    borderSkipped: false,
                }]
            },
            options: {
                ...chartConfig,
                plugins: {
                    ...chartConfig.plugins,
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.1)'
                        },
                        ticks: {
                            font: {
                                family: 'IRANSans, sans-serif'
                            },
                            callback: function(value) {
                                return formatNumber(value);
                            }
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        },
                        ticks: {
                            font: {
                                family: 'IRANSans, sans-serif'
                            }
                        }
                    }
                }
            }
        });
    }

    // Country Chart
    const countryCtx = document.getElementById('countryChart');
    if (countryCtx) {
        countryChart = new Chart(countryCtx.getContext('2d'), {
            type: 'pie',
            data: {
                labels: {!! json_encode(collect($countryStats)->pluck('name')) !!},
                datasets: [{
                    data: {!! json_encode(collect($countryStats)->pluck('count')) !!},
                    backgroundColor: [
                        '#36b9cc', '#1cc88a', '#f6c23e', '#e74a3b', '#6f42c1',
                        '#fd7e14', '#20c997', '#6610f2', '#e83e8c', '#17a2b8'
                    ],
                    borderWidth: 0,
                    hoverBorderWidth: 3,
                    hoverBorderColor: '#fff'
                }]
            },
            options: {
                ...chartConfig,
                plugins: {
                    ...chartConfig.plugins,
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true,
                            font: {
                                family: 'IRANSans, sans-serif'
                            }
                        }
                    }
                }
            }
        });
    }

    // Package Chart
    const packageCtx = document.getElementById('packageChart');
    if (packageCtx) {
        packageChart = new Chart(packageCtx.getContext('2d'), {
            type: 'doughnut',
            data: {
                labels: {!! json_encode(collect($packageStats)->pluck('name')) !!},
                datasets: [{
                    data: {!! json_encode(collect($packageStats)->pluck('count')) !!},
                    backgroundColor: [
                        '#667eea', '#f093fb', '#4facfe', '#43e97b', '#6c757d'
                    ],
                    borderWidth: 0,
                    hoverBorderWidth: 3,
                    hoverBorderColor: '#fff'
                }]
            },
            options: {
                ...chartConfig,
                plugins: {
                    ...chartConfig.plugins,
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true,
                            font: {
                                family: 'IRANSans, sans-serif'
                            }
                        }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = ((context.parsed * 100) / total).toFixed(1);
                                return `${context.label}: ${formatNumber(context.parsed)} (${percentage}%)`;
                            }
                        }
                    }
                },
                cutout: '50%'
            }
        });
    }

    // Monthly Chart
    const monthlyCtx = document.getElementById('monthlyChart');
    if (monthlyCtx) {
        monthlyChart = new Chart(monthlyCtx.getContext('2d'), {
            type: 'line',
            data: {
                labels: {!! json_encode(collect($monthlyStats)->pluck('month_name')) !!},
                datasets: [{
                    label: 'مکان‌های ثبت شده',
                    data: {!! json_encode(collect($monthlyStats)->pluck('count')) !!},
                    borderColor: 'rgba(54, 185, 204, 1)',
                    backgroundColor: 'rgba(54, 185, 204, 0.1)',
                    borderWidth: 3,
                    fill: true,
                    tension: 0.4,
                    pointBackgroundColor: 'rgba(54, 185, 204, 1)',
                    pointBorderColor: '#fff',
                    pointBorderWidth: 2,
                    pointRadius: 6,
                    pointHoverRadius: 8
                }]
            },
            options: {
                ...chartConfig,
                plugins: {
                    ...chartConfig.plugins,
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.1)'
                        },
                        ticks: {
                            font: {
                                family: 'IRANSans, sans-serif'
                            },
                            callback: function(value) {
                                return formatNumber(value);
                            }
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        },
                        ticks: {
                            font: {
                                family: 'IRANSans, sans-serif'
                            }
                        }
                    }
                }
            }
        });
    }
}

// DataTable initialization
function initializeDataTable() {
    const tableElement = document.getElementById('locationsTable');
    if (!tableElement) return;

    if ($.fn.DataTable.isDataTable('#locationsTable')) {
        $('#locationsTable').DataTable().destroy();
    }

    window.locationsTable = $('#locationsTable').DataTable({
        responsive: true,
        pageLength: 25,
        lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "همه"]],
        language: {
            search: "جستجو:",
            lengthMenu: "نمایش _MENU_ ردیف",
            info: "نمایش _START_ تا _END_ از _TOTAL_ ردیف",
            infoEmpty: "نمایش 0 تا 0 از 0 ردیف",
            infoFiltered: "(فیلتر شده از _MAX_ ردیف)",
            paginate: {
                first: "اول",
                last: "آخر",
                next: "بعدی",
                previous: "قبلی"
            },
            emptyTable: "هیچ داده‌ای موجود نیست",
            zeroRecords: "هیچ رکوردی یافت نشد"
        },
        order: [[0, 'desc']],
        columnDefs: [
            { orderable: false, targets: -1 },
            { className: 'text-center', targets: [0, 6, 7, 8] }
        ],
        initComplete: function() {
            console.log('DataTable initialized successfully');
        }
    });
}

// Form interactions
function initializeFormInteractions() {
    // Province-City dependency
    const provinceSelect = document.getElementById('province_id');
    const citySelect = document.getElementById('city_id');

    if (provinceSelect && citySelect) {
        provinceSelect.addEventListener('change', function() {
            const provinceId = this.value;
            showLoading();

            if (provinceId) {
                fetch(`/admin/api/cities/${provinceId}`)
                    .then(response => response.json())
                    .then(cities => {
                        citySelect.innerHTML = '<option value="">همه شهرها</option>';
                        cities.forEach(city => {
                            citySelect.innerHTML += `<option value="${city.id}">${city.name}</option>`;
                        });
                    })
                    .catch(error => {
                        console.error('Error fetching cities:', error);
                        showToast('خطا در دریافت شهرها', 'error');
                    })
                    .finally(() => {
                        hideLoading();
                    });
            } else {
                citySelect.innerHTML = '<option value="">همه شهرها</option>';
                hideLoading();
            }
        });
    }

    // Live search for DataTable
    const searchInput = document.getElementById('search');
    if (searchInput) {
        let searchTimeout;
        searchInput.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(() => {
                if (window.locationsTable) {
                    window.locationsTable.search(this.value).draw();
                }
            }, 500);
        });
    }
}

// Utility functions
function formatNumber(num) {
    return new Intl.NumberFormat('fa-IR').format(num);
}

function getChartConfig() {
    return {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            tooltip: {
                titleFont: {
                    family: 'IRANSans, sans-serif'
                },
                bodyFont: {
                    family: 'IRANSans, sans-serif'
                }
            }
        }
    };
}

function initializeNumberAnimations() {
    // Animate statistics numbers
    const statNumbers = document.querySelectorAll('.stats-number');
    statNumbers.forEach(element => {
        const finalValue = parseInt(element.textContent.replace(/,/g, ''));
        if (!isNaN(finalValue)) {
            animateNumber(element, 0, finalValue, 2000);
        }
    });
}

function animateNumber(element, start, end, duration) {
    const startTime = performance.now();
    const range = end - start;

    function updateNumber(currentTime) {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);
        const current = Math.floor(start + (range * progress));
        element.textContent = formatNumber(current);

        if (progress < 1) {
            requestAnimationFrame(updateNumber);
        }
    }

    requestAnimationFrame(updateNumber);
}

function initializeAutoRefresh() {
    // Auto refresh statistics every 5 minutes
    setInterval(() => {
        fetch(window.location.href + '?ajax=1')
            .then(response => response.json())
            .then(data => {
                if (data.stats) {
                    updateStatistics(data.stats);
                }
            })
            .catch(error => console.error('Auto refresh error:', error));
    }, 300000); // 5 minutes
}

function updateStatistics(stats) {
    document.getElementById('totalLocations').textContent = formatNumber(stats.total);
    document.getElementById('activeLocations').textContent = formatNumber(stats.active);
    document.getElementById('premiumLocations').textContent = formatNumber(stats.premium);
    document.getElementById('todayLocations').textContent = formatNumber(stats.today);
}

function refreshTable() {
    if (window.locationsTable) {
        window.locationsTable.ajax.reload();
    } else {
        location.reload();
    }
}

function exportTableData() {
    // Create CSV export
    const table = document.getElementById('locationsTable');
    const rows = table.querySelectorAll('tr');
    let csv = [];

    for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const cols = row.querySelectorAll('td, th');
        let csvRow = [];

        for (let j = 0; j < cols.length - 1; j++) { // Exclude last column (actions)
            csvRow.push('"' + cols[j].textContent.trim().replace(/"/g, '""') + '"');
        }

        csv.push(csvRow.join(','));
    }

    const csvContent = csv.join('\n');
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', 'locations-report.csv');
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

function showLocationOnMap(lat, lng) {
    const mapUrl = `https://www.google.com/maps?q=${lat},${lng}`;
    window.open(mapUrl, '_blank');
}

// Use functions from admin-reports-locations.js
// All utility functions are defined in the external file
</script>
@endpush