/**
 * Error Fixes for Main Page
 * This file contains fixes for common JavaScript errors
 */

(function() {
    'use strict';

    // Global error handler
    window.addEventListener('error', function(e) {
        console.error('Global error caught:', e.error);
        return false; // Don't prevent default error handling
    });

    // Unhandled promise rejection handler
    window.addEventListener('unhandledrejection', function(e) {
        console.error('Unhandled promise rejection:', e.reason);
    });

    // jQuery availability checker
    function waitForJQuery(callback, maxAttempts = 50) {
        let attempts = 0;
        
        function check() {
            attempts++;
            
            if (typeof window.jQuery !== 'undefined' && typeof window.$ !== 'undefined') {
                console.log('jQuery is available');
                callback();
                return;
            }
            
            if (attempts >= maxAttempts) {
                console.error('jQuery failed to load after', maxAttempts, 'attempts');
                // Try to load jQuery manually
                loadJQueryManually();
                return;
            }
            
            setTimeout(check, 100);
        }
        
        check();
    }

    // Manual jQuery loader
    function loadJQueryManually() {
        console.log('Loading jQuery manually...');
        
        const script = document.createElement('script');
        script.src = 'https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.min.js';
        script.onload = function() {
            console.log('jQuery loaded manually');
            window.$ = window.jQuery;
            initializeAfterJQuery();
        };
        script.onerror = function() {
            console.error('Failed to load jQuery manually');
            // Fallback to local jQuery if available
            loadLocalJQuery();
        };
        document.head.appendChild(script);
    }

    // Local jQuery loader
    function loadLocalJQuery() {
        const script = document.createElement('script');
        script.src = '/js/jquery-3.6.0.min.js';
        script.onload = function() {
            console.log('Local jQuery loaded');
            window.$ = window.jQuery;
            initializeAfterJQuery();
        };
        script.onerror = function() {
            console.error('Local jQuery also failed to load');
        };
        document.head.appendChild(script);
    }

    // Initialize components after jQuery is available
    function initializeAfterJQuery() {
        console.log('Initializing components after jQuery...');
        
        // Wait for DOM to be ready
        $(document).ready(function() {
            console.log('DOM ready with jQuery available');
            
            // Initialize Select2 safely
            initializeSelect2Safe();
            
            // Initialize other jQuery-dependent components
            initializeJQueryComponents();
        });
    }

    // Safe Select2 initialization
    function initializeSelect2Safe() {
        if (typeof $.fn.select2 === 'undefined') {
            console.log('Select2 not available, loading...');
            loadSelect2();
            return;
        }
        
        console.log('Initializing Select2...');
        
        // Initialize country select if it exists
        const countrySelect = $('#country-select');
        if (countrySelect.length && !countrySelect.hasClass('select2-hidden-accessible')) {
            try {
                countrySelect.select2({
                    width: '100%',
                    placeholder: 'انتخاب کشور...',
                    allowClear: true
                });
                console.log('Country select initialized');
            } catch (error) {
                console.error('Error initializing country select:', error);
            }
        }
    }

    // Load Select2
    function loadSelect2() {
        const script = document.createElement('script');
        script.src = 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js';
        script.onload = function() {
            console.log('Select2 loaded');
            setTimeout(initializeSelect2Safe, 100);
        };
        script.onerror = function() {
            console.error('Failed to load Select2');
        };
        document.head.appendChild(script);
    }

    // Initialize other jQuery components
    function initializeJQueryComponents() {
        // Fix resetAllFilters function
        window.resetAllFilters = function() {
            console.log('Resetting all filters safely...');
            
            try {
                // Reset search input
                const searchInput = document.getElementById('search-input');
                if (searchInput) {
                    searchInput.value = '';
                }
                
                // Reset country select
                const countrySelect = $('#country-select');
                if (countrySelect.length) {
                    countrySelect.val('').trigger('change');
                }
                
                // Reset other selects
                const selects = ['province-select', 'county-select', 'district-select', 'city-select', 'village-select', 'zone-select'];
                selects.forEach(function(selectId) {
                    const select = $('#' + selectId);
                    if (select.length) {
                        select.val('').trigger('change');
                        select.closest('.mb-3').hide();
                    }
                });
                
                // Reset category filters
                $('.parent-category.active, .child-category.active').removeClass('active');
                
                console.log('All filters reset successfully');
            } catch (error) {
                console.error('Error resetting filters:', error);
            }
        };
        
        // Fix clear filters button
        const clearFiltersBtn = document.getElementById('clear-filters');
        if (clearFiltersBtn) {
            clearFiltersBtn.addEventListener('click', function(e) {
                e.preventDefault();
                if (typeof window.resetAllFilters === 'function') {
                    window.resetAllFilters();
                }
            });
        }
        
        // Fix reset search button
        const resetSearchBtn = document.getElementById('reset-search');
        if (resetSearchBtn) {
            resetSearchBtn.addEventListener('click', function(e) {
                e.preventDefault();
                if (typeof window.resetAllFilters === 'function') {
                    window.resetAllFilters();
                }
                if (typeof window.loadLocations === 'function') {
                    window.loadLocations();
                }
            });
        }
    }

    // Map initialization fix
    function initializeMapSafely() {
        // Check if Leaflet is available
        if (typeof L === 'undefined') {
            console.log('Leaflet not available, waiting...');
            setTimeout(initializeMapSafely, 500);
            return;
        }
        
        // Check if map container exists
        const mapContainer = document.getElementById('map');
        if (!mapContainer) {
            // Only log if we're on a page that should have a map
            if (window.location.pathname.includes('/map') ||
                document.querySelector('[data-map-required]')) {
                console.log('Map container not found on page that requires map');
            }
            return;
        }

        // Skip map initialization on map page - let MapManager handle it
        if (window.location.pathname.includes('/map')) {
            console.log('On map page, skipping error-fixes map initialization');
            return;
        }

        // Check if MapManager is handling the map
        if (window.mapController) {
            console.log('MapManager is handling the map, skipping error-fixes map initialization');
            return;
        }

        // Check if map is already initialized
        if (window.map && typeof window.map.getZoom === 'function') {
            console.log('Map already initialized');
            return;
        }
        
        try {
            console.log('Initializing map safely...');
            window.map = L.map('map').setView([32.4279, 53.6880], 5);
            
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
            }).addTo(window.map);
            
            console.log('Map initialized successfully');
        } catch (error) {
            console.error('Error initializing map:', error);
        }
    }

    // Initialize everything when DOM is loaded
    document.addEventListener('DOMContentLoaded', function() {
        console.log('DOM loaded, starting error fixes...');
        
        // Wait for jQuery and initialize
        waitForJQuery(function() {
            initializeAfterJQuery();
        });
        
        // Initialize map
        setTimeout(initializeMapSafely, 1000);
    });

    // Listen for scripts loaded event
    window.addEventListener('scriptsLoaded', function(e) {
        console.log('Scripts loaded event received:', e.detail);
        
        // Re-initialize components
        setTimeout(function() {
            if (typeof $ !== 'undefined') {
                initializeAfterJQuery();
            }
            initializeMapSafely();
        }, 500);
    });

})();
