/**
 * فایل JavaScript پنل مدیریت
 * شامل توابع مورد نیاز برای تعاملات پنل
 */

// تنظیمات اولیه
document.addEventListener('DOMContentLoaded', function() {
    // راه‌اندازی منوی موبایل
    initMobileMenu();
    
    // راه‌اندازی مودال‌ها
    initModals();
    
    // راه‌اندازی تب‌ها
    initTabs();
    
    // راه‌اندازی آپلود فایل
    initFileUpload();
    
    // راه‌اندازی نمودارها
    initCharts();
    
    // راه‌اندازی دیتاتیبل
    initDataTable();
    
    // راه‌اندازی انتخابگر تاریخ
    initDatePickers();
    
    // راه‌اندازی ویرایشگر متن
    initTextEditor();
});

/**
 * مدیریت منوی موبایل
 */
function initMobileMenu() {
    const menuToggle = document.querySelector('.admin-menu-toggle');
    const sidebar = document.querySelector('.admin-sidebar');
    
    if (menuToggle && sidebar) {
        menuToggle.addEventListener('click', function() {
            sidebar.classList.toggle('open');
        });
        
        // بستن منو با کلیک خارج از آن
        document.addEventListener('click', function(e) {
            if (!sidebar.contains(e.target) && !menuToggle.contains(e.target)) {
                sidebar.classList.remove('open');
            }
        });
    }
}

/**
 * مدیریت مودال‌ها
 */
function initModals() {
    // باز کردن مودال
    document.querySelectorAll('[data-modal]').forEach(trigger => {
        trigger.addEventListener('click', function() {
            const modalId = this.dataset.modal;
            const modal = document.getElementById(modalId);
            if (modal) {
                modal.classList.add('show');
                document.body.style.overflow = 'hidden';
            }
        });
    });
    
    // بستن مودال
    document.querySelectorAll('.admin-modal-close, .admin-modal').forEach(element => {
        element.addEventListener('click', function(e) {
            if (e.target === this) {
                const modal = this.closest('.admin-modal');
                if (modal) {
                    modal.classList.remove('show');
                    document.body.style.overflow = '';
                }
            }
        });
    });
}

/**
 * مدیریت تب‌ها
 */
function initTabs() {
    document.querySelectorAll('.admin-tab-item').forEach(tab => {
        tab.addEventListener('click', function() {
            const tabId = this.dataset.tab;
            const tabGroup = this.closest('.admin-tabs');
            
            // حذف کلاس active از همه تب‌ها
            tabGroup.querySelectorAll('.admin-tab-item').forEach(t => {
                t.classList.remove('active');
            });
            
            // مخفی کردن همه محتواها
            tabGroup.querySelectorAll('.admin-tab-content').forEach(c => {
                c.classList.remove('active');
            });
            
            // فعال کردن تب انتخاب شده
            this.classList.add('active');
            document.getElementById(tabId).classList.add('active');
        });
    });
}

/**
 * مدیریت آپلود فایل
 */
function initFileUpload() {
    document.querySelectorAll('.admin-upload').forEach(upload => {
        const input = upload.querySelector('input[type="file"]');
        const preview = upload.querySelector('.admin-upload-preview');
        
        if (input && preview) {
            // نمایش پیش‌نمایش فایل‌ها
            input.addEventListener('change', function() {
                preview.innerHTML = '';
                
                Array.from(this.files).forEach(file => {
                    if (file.type.startsWith('image/')) {
                        const reader = new FileReader();
                        reader.onload = function(e) {
                            const item = document.createElement('div');
                            item.className = 'admin-upload-item';
                            item.innerHTML = `
                                <img src="${e.target.result}" alt="${file.name}">
                                <button type="button" class="admin-upload-remove">
                                    <i class="fas fa-times"></i>
                                </button>
                            `;
                            preview.appendChild(item);
                        };
                        reader.readAsDataURL(file);
                    }
                });
            });
            
            // حذف فایل از پیش‌نمایش
            preview.addEventListener('click', function(e) {
                if (e.target.closest('.admin-upload-remove')) {
                    e.target.closest('.admin-upload-item').remove();
                }
            });
        }
    });
}

/**
 * مدیریت نمودارها
 */
function initCharts() {
    document.querySelectorAll('.admin-chart').forEach(chart => {
        const ctx = chart.getContext('2d');
        const type = chart.dataset.type || 'line';
        const data = JSON.parse(chart.dataset.chart || '{}');
        
        new Chart(ctx, {
            type: type,
            data: data,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                    }
                }
            }
        });
    });
}

/**
 * مدیریت دیتاتیبل
 */
function initDataTable() {
    document.querySelectorAll('.admin-table').forEach(table => {
        if (table.classList.contains('datatable')) {
            new DataTable(table, {
                language: {
                    url: '/assets/js/datatable-persian.json'
                },
                responsive: true,
                pageLength: 10,
                order: [[0, 'desc']]
            });
        }
    });
}

/**
 * مدیریت انتخابگر تاریخ
 */
function initDatePickers() {
    // استفاده از LocaleDateManager برای تنظیم تقویم‌ها
    if (window.localeDateManager) {
        window.localeDateManager.updateDatePickers();
    } else {
        // Fallback: تنظیم دستی
        document.querySelectorAll('.admin-datepicker').forEach(input => {
            const locale = document.documentElement.lang || 'fa';

            if (locale === 'fa') {
                new Pikaday({
                    field: input,
                    format: 'YYYY/MM/DD',
                    i18n: {
                        previousMonth: 'ماه قبل',
                        nextMonth: 'ماه بعد',
                        months: ['فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور', 'مهر', 'آبان', 'آذر', 'دی', 'بهمن', 'اسفند'],
                        weekdays: ['یکشنبه', 'دوشنبه', 'سه‌شنبه', 'چهارشنبه', 'پنج‌شنبه', 'جمعه', 'شنبه'],
                        weekdaysShort: ['یک', 'دو', 'سه', 'چهار', 'پنج', 'جم', 'شن']
                    }
                });
            } else {
                new Pikaday({
                    field: input,
                    format: 'YYYY-MM-DD'
                });
            }
        });
    }
}

/**
 * مدیریت ویرایشگر متن
 */
function initTextEditor() {
    document.querySelectorAll('.admin-editor').forEach(editor => {
        ClassicEditor
            .create(editor, {
                language: 'fa',
                toolbar: ['heading', '|', 'bold', 'italic', 'link', 'bulletedList', 'numberedList', '|', 'outdent', 'indent', '|', 'blockQuote', 'insertTable', 'undo', 'redo']
            })
            .catch(error => {
                console.error(error);
            });
    });
}

/**
 * نمایش پیام‌های هشدار
 */
function showAlert(message, type = 'success') {
    const alert = document.createElement('div');
    alert.className = `admin-alert admin-alert-${type}`;
    alert.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'danger' ? 'times-circle' : type === 'warning' ? 'exclamation-circle' : 'info-circle'}"></i>
        <span>${message}</span>
    `;
    
    document.querySelector('.admin-content').insertAdjacentElement('afterbegin', alert);
    
    setTimeout(() => {
        alert.remove();
    }, 5000);
}

/**
 * نمایش پیش‌بارگذاری
 */
function showLoading() {
    const loading = document.createElement('div');
    loading.className = 'admin-loading';
    loading.innerHTML = '<div class="admin-spinner"></div>';
    document.body.appendChild(loading);
}

function hideLoading() {
    const loading = document.querySelector('.admin-loading');
    if (loading) {
        loading.remove();
    }
}

/**
 * ارسال درخواست AJAX
 */
function ajaxRequest(url, method = 'GET', data = null) {
    showLoading();
    
    return fetch(url, {
        method: method,
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        },
        body: data ? JSON.stringify(data) : null
    })
    .then(response => response.json())
    .finally(() => {
        hideLoading();
    });
}

/**
 * اعتبارسنجی فرم
 */
function validateForm(form) {
    let isValid = true;
    const inputs = form.querySelectorAll('input, select, textarea');
    
    inputs.forEach(input => {
        if (input.hasAttribute('required') && !input.value.trim()) {
            isValid = false;
            input.classList.add('is-invalid');
        } else {
            input.classList.remove('is-invalid');
        }
    });
    
    return isValid;
}

/**
 * تبدیل تاریخ میلادی به شمسی
 */
function toJalali(date) {
    if (window.localeDateManager) {
        return window.localeDateManager.toJalali(date);
    }
    // Fallback
    return moment(date).format('jYYYY/jMM/jDD');
}

/**
 * تبدیل تاریخ شمسی به میلادی
 */
function toGregorian(date) {
    if (window.localeDateManager) {
        return window.localeDateManager.toGregorian(date);
    }
    // Fallback
    return moment(date, 'jYYYY/jMM/jDD').format('YYYY-MM-DD');
}

/**
 * فرمت‌بندی اعداد بر اساس زبان
 */
function formatNumber(number) {
    const locale = document.documentElement.lang || 'fa';
    const formatted = new Intl.NumberFormat(locale === 'fa' ? 'fa-IR' : 'en-US').format(number);

    if (window.localeDateManager && locale === 'fa') {
        return window.localeDateManager.toPersianNumbers(formatted);
    }

    return formatted;
}

/**
 * فرمت‌بندی پول
 */
function formatMoney(amount) {
    const locale = document.documentElement.lang || 'fa';
    const currency = locale === 'fa' ? ' تومان' : ' USD';
    return formatNumber(amount) + currency;
}

/**
 * فرمت‌بندی تاریخ بر اساس زبان
 */
function formatDate(date) {
    if (window.localeDateManager) {
        return window.localeDateManager.formatDateForChart(date);
    }
    // Fallback
    const locale = document.documentElement.lang || 'fa';
    if (locale === 'fa') {
        return moment(date).format('jYYYY/jMM/jDD HH:mm');
    } else {
        return new Date(date).toLocaleDateString(locale) + ' ' + new Date(date).toLocaleTimeString(locale);
    }
}

/**
 * کپی متن به کلیپ‌بورد
 */
function copyToClipboard(text) {
    const textarea = document.createElement('textarea');
    textarea.value = text;
    document.body.appendChild(textarea);
    textarea.select();
    document.execCommand('copy');
    document.body.removeChild(textarea);
    showAlert('متن با موفقیت کپی شد');
}

/**
 * دانلود فایل
 */
function downloadFile(url, filename) {
    const link = document.createElement('a');
    link.href = url;
    link.download = filename;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

/**
 * چاپ صفحه
 */
function printPage() {
    window.print();
}

/**
 * تغییر تم
 */
function toggleTheme() {
    document.body.classList.toggle('dark-theme');
    localStorage.setItem('theme', document.body.classList.contains('dark-theme') ? 'dark' : 'light');
}

// بررسی تم ذخیره شده
if (localStorage.getItem('theme') === 'dark') {
    document.body.classList.add('dark-theme');
} 