<?php

namespace App\Services;

use App\Models\Language;
use Illuminate\Support\Facades\DB;

/**
 * سرویس مدیریت ترجمه‌های دیتابیسی
 * این سرویس عملیات CRUD و ابزارهای کمکی ترجمه را انجام می‌دهد.
 *
 * @author AI
 */
class TranslationService
{
    /**
     * دریافت ترجمه‌ها با فیلتر و صفحه‌بندی
     */
    public function getPaginatedTranslations($filters = [], $perPage = 20)
    {
        $query = DB::table('translations');
        if (! empty($filters['search'])) {
            $query->where(function ($q) use ($filters) {
                $q->where('key', 'like', '%'.$filters['search'].'%')
                    ->orWhere('value', 'like', '%'.$filters['search'].'%');
            });
        }
        if (! empty($filters['language'])) {
            $query->where('language_code', $filters['language']);
        }
        if (! empty($filters['group'])) {
            $query->where('group', $filters['group']);
        }
        if (! empty($filters['namespace'])) {
            $query->where('namespace', $filters['namespace']);
        }
        $query->orderByDesc('updated_at');

        return $query->paginate($perPage);
    }

    /**
     * دریافت گروه‌های ترجمه
     */
    public function getGroups()
    {
        return DB::table('translations')->distinct()->pluck('group')->filter()->values();
    }

    /**
     * دریافت namespaceهای ترجمه
     */
    public function getNamespaces()
    {
        return DB::table('translations')->distinct()->pluck('namespace')->filter()->values();
    }

    /**
     * ایجاد ترجمه جدید
     */
    public function createTranslation($data)
    {
        $data['group'] = $data['group'] ?? '*';
        $data['namespace'] = $data['namespace'] ?? '*';
        // اگر مقدار value آرایه بود، آن را به رشته JSON تبدیل کن (برای جلوگیری از خطای دیتابیس)
        if (is_array($data['value'])) {
            $data['value'] = json_encode($data['value'], JSON_UNESCAPED_UNICODE);
        }
        DB::table('translations')->updateOrInsert([
            'key' => $data['key'],
            'language_code' => $data['language_code'],
            'group' => $data['group'],
            'namespace' => $data['namespace'],
        ], [
            'value' => $data['value'],
            'updated_at' => now(),
            'created_at' => now(),
        ]);
    }

    /**
     * پیدا کردن ترجمه بر اساس آیدی
     */
    public function findTranslation($id)
    {
        $row = DB::table('translations')->where('id', $id)->first();
        if (! $row) {
            abort(404, 'ترجمه یافت نشد.');
        }
        $row->language = Language::where('code', $row->language_code)->first();

        return $row;
    }

    /**
     * بروزرسانی ترجمه
     */
    public function updateTranslation($id, $data)
    {
        DB::table('translations')->where('id', $id)->update([
            'value' => $data['value'],
            'updated_at' => now(),
        ]);
    }

    /**
     * حذف ترجمه
     */
    public function deleteTranslation($id)
    {
        DB::table('translations')->where('id', $id)->delete();
    }

    /**
     * ایمپورت ترجمه‌ها از فایل‌های زبان (نمونه ساده)
     */
    public function importFromFiles()
    {
        // فقط یک نمونه ساده: فایل lang/fa/menu.php و lang/en/menu.php
        $languages = Language::where('is_active', 1)->pluck('code');
        foreach ($languages as $lang) {
            $path = resource_path("lang/{$lang}/menu.php");
            if (file_exists($path)) {
                $array = include $path;
                foreach ($array as $key => $value) {
                    $this->createTranslation([
                        'key' => $key,
                        'value' => $value,
                        'language_code' => $lang,
                        'group' => 'menu',
                        'namespace' => '*',
                    ]);
                }
            }
        }
    }

    /**
     * اکسپورت ترجمه‌ها به فایل CSV
     */
    public function exportToFile()
    {
        $rows = DB::table('translations')->get();
        $filename = 'translations_export_'.now()->format('Ymd_His').'.csv';
        $handle = fopen(storage_path('app/'.$filename), 'w+');
        fputcsv($handle, ['id', 'key', 'value', 'language_code', 'group', 'namespace']);
        foreach ($rows as $row) {
            fputcsv($handle, [
                $row->id, $row->key, $row->value, $row->language_code, $row->group, $row->namespace,
            ]);
        }
        fclose($handle);

        return response()->download(storage_path('app/'.$filename))->deleteFileAfterSend(true);
    }

    /**
     * اسکن فایل‌های زبان برای کلیدهای جدید (نمونه ساده)
     */
    public function scanForNewKeys()
    {
        // فقط یک نمونه ساده: اسکن فایل lang/fa/menu.php
        $path = resource_path('lang/fa/menu.php');
        if (file_exists($path)) {
            $array = include $path;
            foreach ($array as $key => $value) {
                $exists = DB::table('translations')->where('key', $key)->where('language_code', 'fa')->exists();
                if (! $exists) {
                    $this->createTranslation([
                        'key' => $key,
                        'value' => $value,
                        'language_code' => 'fa',
                        'group' => 'menu',
                        'namespace' => '*',
                    ]);
                }
            }
        }
    }
}
