<?php

namespace App\Services;

use App\Models\SiteMenu;
use App\Models\HeaderSetting;
use App\Models\ButtonSetting;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Auth;

class MenuService
{
    /**
     * Get site menus for display.
     */
    public function getSiteMenus($languageCode = 'fa', $useCache = true)
    {
        $cacheKey = 'site_menus_' . $languageCode;
        
        if (!$useCache) {
            Cache::forget($cacheKey);
        }

        return Cache::remember($cacheKey, 3600, function () use ($languageCode) {
            $isGuest = !Auth::check();
            return SiteMenu::getMenuTree($languageCode, $isGuest);
        });
    }

    /**
     * Get header settings.
     */
    public function getHeaderSettings($languageCode = 'fa', $useCache = true)
    {
        $cacheKey = 'header_settings_' . $languageCode;
        
        if (!$useCache) {
            Cache::forget($cacheKey);
        }

        return Cache::remember($cacheKey, 3600, function () use ($languageCode) {
            return HeaderSetting::getAllForLanguage($languageCode);
        });
    }

    /**
     * Get button settings.
     */
    public function getButtonSettings($languageCode = 'fa', $useCache = true)
    {
        $cacheKey = 'button_settings_' . $languageCode;
        
        if (!$useCache) {
            Cache::forget($cacheKey);
        }

        return Cache::remember($cacheKey, 3600, function () use ($languageCode) {
            $isGuest = !Auth::check();
            return ButtonSetting::getButtonsForUser($languageCode, $isGuest);
        });
    }

    /**
     * Get complete header data for templates.
     */
    public function getHeaderData($languageCode = 'fa')
    {
        return [
            'menus' => $this->getSiteMenus($languageCode),
            'header_settings' => $this->getHeaderSettings($languageCode),
            'buttons' => $this->getButtonSettings($languageCode),
            'is_guest' => !Auth::check(),
            'current_user' => Auth::user(),
        ];
    }

    /**
     * Check if search box should be shown.
     */
    public function shouldShowSearchBox($languageCode = 'fa')
    {
        return HeaderSetting::getBoolValue('show_search_box', $languageCode, true);
    }

    /**
     * Get search placeholder text.
     */
    public function getSearchPlaceholder($languageCode = 'fa')
    {
        return HeaderSetting::getValue('search_placeholder', $languageCode, 'جستجو...');
    }

    /**
     * Check if language switcher should be shown.
     */
    public function shouldShowLanguageSwitcher($languageCode = 'fa')
    {
        return HeaderSetting::getBoolValue('show_language_switcher', $languageCode, true);
    }

    /**
     * Check if user menu should be shown.
     */
    public function shouldShowUserMenu($languageCode = 'fa')
    {
        return HeaderSetting::getBoolValue('show_user_menu', $languageCode, true);
    }

    /**
     * Get header style.
     */
    public function getHeaderStyle($languageCode = 'fa')
    {
        return HeaderSetting::getValue('header_style', $languageCode, 'default');
    }

    /**
     * Check if header should be sticky.
     */
    public function isStickyHeader($languageCode = 'fa')
    {
        return HeaderSetting::getBoolValue('sticky_header', $languageCode, true);
    }

    /**
     * Check if breadcrumb should be shown.
     */
    public function shouldShowBreadcrumb($languageCode = 'fa')
    {
        return HeaderSetting::getBoolValue('show_breadcrumb', $languageCode, true);
    }

    /**
     * Get specific button by key.
     */
    public function getButton($key, $languageCode = 'fa')
    {
        return ButtonSetting::getButton($key, $languageCode);
    }

    /**
     * Clear all menu-related caches.
     */
    public function clearCache($languageCode = null)
    {
        if ($languageCode) {
            Cache::forget('site_menus_' . $languageCode);
            Cache::forget('header_settings_' . $languageCode);
            Cache::forget('button_settings_' . $languageCode);
        } else {
            // Clear for all languages
            try {
                $languages = \App\Models\Language::active()->pluck('code');
                foreach ($languages as $code) {
                    Cache::forget('site_menus_' . $code);
                    Cache::forget('header_settings_' . $code);
                    Cache::forget('button_settings_' . $code);
                }
            } catch (\Exception $e) {
                // Fallback: clear common language codes
                $commonLanguages = ['fa', 'en'];
                foreach ($commonLanguages as $code) {
                    Cache::forget('site_menus_' . $code);
                    Cache::forget('header_settings_' . $code);
                    Cache::forget('button_settings_' . $code);
                }
            }
        }
    }

    /**
     * Generate breadcrumb for current page.
     */
    public function generateBreadcrumb($currentPage = null, $languageCode = 'fa')
    {
        $breadcrumb = [
            [
                'title' => 'صفحه اصلی',
                'url' => '/',
                'active' => false
            ]
        ];

        if ($currentPage) {
            $breadcrumb[] = [
                'title' => $currentPage,
                'url' => null,
                'active' => true
            ];
        }

        return $breadcrumb;
    }

    /**
     * Get menu statistics for admin dashboard.
     */
    public function getMenuStatistics()
    {
        return [
            'total_menus' => SiteMenu::count(),
            'active_menus' => SiteMenu::active()->count(),
            'main_menus' => SiteMenu::main()->count(),
            'sub_menus' => SiteMenu::whereNotNull('parent_id')->count(),
        ];
    }
}
