<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\Pivot;

class ZoneAd extends Pivot
{
    use HasFactory;

    /**
     * Indicates if the IDs are auto-incrementing.
     *
     * @var bool
     */
    public $incrementing = true;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'zone_ads';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'zone_id',
        'ad_id',
        'location_id',
        'display_order',
        'is_active',
        'start_date',
        'end_date',
        'created_by',
        'updated_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'is_active' => 'boolean',
        'display_order' => 'integer',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
    ];

    /**
     * رابطه با زون نمایش
     */
    public function zone()
    {
        return $this->belongsTo(DisplayZone::class, 'zone_id');
    }

    /**
     * رابطه با آگهی
     */
    public function ad()
    {
        return $this->belongsTo(Ad::class, 'ad_id');
    }

    /**
     * رابطه با موقعیت (شهر/استان)
     */
    public function location()
    {
        return $this->belongsTo(Location::class, 'location_id');
    }

    /**
     * رابطه با کاربر ایجاد کننده
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * رابطه با کاربر ویرایش کننده
     */
    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * فیلتر کردن رکوردهای فعال
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('start_date')
                    ->orWhere('start_date', '<=', now());
            })
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            });
    }

    /**
     * ثبت بازدید از آگهی
     */
    public function recordView()
    {
        if ($this->ad) {
            $this->ad->incrementViewCount();
        }
    }

    /**
     * ثبت کلیک روی آگهی
     */
    public function recordClick()
    {
        if ($this->ad) {
            $this->ad->incrementClickCount();
        }
    }
}
