<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class HeaderSetting extends Model
{
    use HasFactory;

    protected $table = 'header_settings';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'setting_key',
        'setting_value',
        'language_code',
        'description',
        'is_active',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'is_active' => 'boolean',
    ];

    /**
     * Get the language.
     */
    public function language()
    {
        return $this->belongsTo(Language::class, 'language_code', 'code');
    }

    /**
     * Scope a query to only include active settings.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to filter by language.
     */
    public function scopeForLanguage($query, $languageCode)
    {
        return $query->where('language_code', $languageCode);
    }

    /**
     * Get setting value by key and language.
     */
    public static function getValue($key, $languageCode = 'fa', $default = null)
    {
        $setting = self::active()
            ->forLanguage($languageCode)
            ->where('setting_key', $key)
            ->first();

        return $setting ? $setting->setting_value : $default;
    }

    /**
     * Set setting value by key and language.
     */
    public static function setValue($key, $value, $languageCode = 'fa', $description = null)
    {
        return self::updateOrCreate(
            [
                'setting_key' => $key,
                'language_code' => $languageCode,
            ],
            [
                'setting_value' => $value,
                'description' => $description,
                'is_active' => true,
            ]
        );
    }

    /**
     * Get all settings for a language as key-value array.
     */
    public static function getAllForLanguage($languageCode = 'fa')
    {
        return self::active()
            ->forLanguage($languageCode)
            ->pluck('setting_value', 'setting_key')
            ->toArray();
    }

    /**
     * Get boolean value for setting.
     */
    public static function getBoolValue($key, $languageCode = 'fa', $default = false)
    {
        $value = self::getValue($key, $languageCode, $default);
        return filter_var($value, FILTER_VALIDATE_BOOLEAN);
    }
}
