<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\Response;

class XssProtection
{
    /**
     * الگوهای مخرب XSS
     */
    protected $xssPatterns = [
        // اسکریپت‌های مخرب
        '/<script\b[^>]*>(.*?)<\/script>/is',
        '/javascript:/i',
        '/vbscript:/i',
        '/onload=/i',
        '/onerror=/i',
        '/onmouseover=/i',
        '/onclick=/i',
        '/onfocus=/i',
        '/onblur=/i',
        '/onchange=/i',
        '/onsubmit=/i',
        '/onreset=/i',
        '/onselect=/i',
        '/onkeydown=/i',
        '/onkeypress=/i',
        '/onkeyup=/i',
        '/onmousedown=/i',
        '/onmousemove=/i',
        '/onmouseout=/i',
        '/onmouseup=/i',
        '/onunload=/i',

        // پروتکل‌های مخرب
        '/data:/i',
        '/vbscript:/i',
        '/file:/i',
        '/about:/i',
        '/telnet:/i',
        '/gopher:/i',
        '/news:/i',
        '/nntp:/i',
        '/feed:/i',
        '/ftp:/i',

        // تگ‌های مخرب
        '/<iframe/i',
        '/<frame/i',
        '/<object/i',
        '/<embed/i',
        '/<applet/i',
        '/<meta/i',
        '/<link/i',
        '/<style/i',
        '/<base/i',
        '/<form/i',
        '/<input/i',
        '/<textarea/i',
        '/<select/i',
        '/<button/i',
        '/<img/i',
        '/<svg/i',
        '/<math/i',
        '/<video/i',
        '/<audio/i',
        '/<source/i',
        '/<track/i',
        '/<canvas/i',
        '/<map/i',
        '/<area/i',
        '/<details/i',
        '/<dialog/i',
        '/<menu/i',
        '/<menuitem/i',
        '/<summary/i',
        '/<template/i',
        '/<slot/i',
        '/<shadow/i',
        '/<part/i',
        '/<content/i',
        '/<element/i',
        '/<decorator/i',
        '/<import/i',
        '/<include/i',
        '/<use/i',
        '/<defs/i',
        '/<desc/i',
        '/<title/i',
        '/<foreignObject/i',
        '/<image/i',
        '/<script/i',
        '/<style/i',
        '/<link/i',
        '/<meta/i',
        '/<base/i',
        '/<form/i',
        '/<input/i',
        '/<textarea/i',
        '/<select/i',
        '/<button/i',
        '/<img/i',
        '/<svg/i',
        '/<math/i',
        '/<video/i',
        '/<audio/i',
        '/<source/i',
        '/<track/i',
        '/<canvas/i',
        '/<map/i',
        '/<area/i',
        '/<details/i',
        '/<dialog/i',
        '/<menu/i',
        '/<menuitem/i',
        '/<summary/i',
        '/<template/i',
        '/<slot/i',
        '/<shadow/i',
        '/<part/i',
        '/<content/i',
        '/<element/i',
        '/<decorator/i',
        '/<import/i',
        '/<include/i',
        '/<use/i',
        '/<defs/i',
        '/<desc/i',
        '/<title/i',
        '/<foreignObject/i',
        '/<image/i',
    ];

    public function handle(Request $request, Closure $next): Response
    {
        // بررسی داده‌های GET
        if ($this->containsXss($request->query())) {
            $this->logXssAttempt($request, 'GET');

            return response()->json([
                'message' => 'داده‌های ارسالی حاوی کد مخرب است.',
                'error_code' => 'xss_detected',
            ], 400);
        }

        // بررسی داده‌های POST
        if ($this->containsXss($request->post())) {
            $this->logXssAttempt($request, 'POST');

            return response()->json([
                'message' => 'داده‌های ارسالی حاوی کد مخرب است.',
                'error_code' => 'xss_detected',
            ], 400);
        }

        // بررسی داده‌های JSON
        if ($request->isJson() && $this->containsXss($request->json()->all())) {
            $this->logXssAttempt($request, 'JSON');

            return response()->json([
                'message' => 'داده‌های ارسالی حاوی کد مخرب است.',
                'error_code' => 'xss_detected',
            ], 400);
        }

        // بررسی هدرها
        if ($this->containsXss($request->headers->all())) {
            $this->logXssAttempt($request, 'HEADER');

            return response()->json([
                'message' => 'هدرهای درخواست حاوی کد مخرب است.',
                'error_code' => 'xss_detected',
            ], 400);
        }

        $response = $next($request);

        // تنظیم هدرهای امنیتی
        $this->setSecurityHeaders($response);

        return $response;
    }

    protected function containsXss($data): bool
    {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                if (is_array($value)) {
                    if ($this->containsXss($value)) {
                        return true;
                    }
                } else {
                    if ($this->checkXssPatterns($value)) {
                        return true;
                    }
                }
            }
        } else {
            return $this->checkXssPatterns($data);
        }

        return false;
    }

    protected function checkXssPatterns($value): bool
    {
        if (! is_string($value)) {
            return false;
        }

        foreach ($this->xssPatterns as $pattern) {
            if (preg_match($pattern, $value)) {
                return true;
            }
        }

        return false;
    }

    protected function setSecurityHeaders(Response $response): void
    {
        $response->headers->set('X-XSS-Protection', '1; mode=block');
        $response->headers->set('Content-Security-Policy', "default-src 'self'; script-src 'self' 'unsafe-inline' 'unsafe-eval'; style-src 'self' 'unsafe-inline'; img-src 'self' data: https:; font-src 'self' data: https:; connect-src 'self'");
        $response->headers->set('X-Content-Type-Options', 'nosniff');
        $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');
    }

    protected function logXssAttempt(Request $request, string $type): void
    {
        Log::warning('تلاش XSS شناسایی شد', [
            'ip' => $request->ip(),
            'method' => $request->method(),
            'url' => $request->fullUrl(),
            'user_agent' => $request->userAgent(),
            'type' => $type,
            'data' => $request->all(),
        ]);
    }
}
