<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CategoryTemplate;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TemplateController extends Controller
{
    /**
     * نمایش لیست قالب‌ها
     */
    public function index()
    {
        $templates = CategoryTemplate::with('category')->get();
        return view('admin.templates.index', compact('templates'));
    }

    /**
     * نمایش فرم ایجاد قالب جدید
     */
    public function create()
    {
        $categories = Category::all();
        $availableFeatures = config('features.available', []);
        return view('admin.templates.create', compact('categories', 'availableFeatures'));
    }

    /**
     * ذخیره قالب جدید
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'template_key' => 'required|string|max:50|unique:category_templates,template_key',
            'template_name' => 'required|string|max:100',
            'category_id' => 'required|exists:categories,id',
            'header_style' => 'required|string|max:50',
            'content_layout' => 'required|string|max:50',
            'primary_color' => 'required|string|max:20',
            'secondary_color' => 'required|string|max:20',
            'icon_set' => 'required|string|max:50',
            'template_type' => 'required|in:base,parent,child,custom',
            'display_order' => 'nullable|integer',
            'is_active' => 'nullable|boolean',
            'features' => 'nullable|array',
            'custom_fields' => 'nullable|array',
        ]);

        $validated['features'] = $validated['features'] ?? [];
        $validated['custom_fields'] = $validated['custom_fields'] ?? [];
        $validated['is_active'] = $request->has('is_active') ? 1 : 0;
        $validated['display_order'] = $validated['display_order'] ?? 10;

        CategoryTemplate::create($validated);

        return redirect()->route('admin.templates.index')
            ->with('success', 'قالب با موفقیت ایجاد شد.');
    }

    /**
     * نمایش فرم ویرایش قالب
     */
    public function edit(CategoryTemplate $template)
    {
        $categories = Category::all();
        $availableFeatures = config('features.available', []);
        return view('admin.templates.edit', compact('template', 'categories', 'availableFeatures'));
    }

    /**
     * به‌روزرسانی قالب
     */
    public function update(Request $request, CategoryTemplate $template)
    {
        $validated = $request->validate([
            'template_key' => 'required|string|max:50|unique:category_templates,template_key,'.$template->id,
            'template_name' => 'required|string|max:100',
            'category_id' => 'required|exists:categories,id',
            'header_style' => 'required|string|max:50',
            'content_layout' => 'required|string|max:50',
            'primary_color' => 'required|string|max:20',
            'secondary_color' => 'required|string|max:20',
            'icon_set' => 'required|string|max:50',
            'template_type' => 'required|in:base,parent,child,custom',
            'display_order' => 'nullable|integer',
            'is_active' => 'nullable|boolean',
            'features' => 'nullable|array',
            'custom_fields' => 'nullable|array',
        ]);

        $validated['features'] = $validated['features'] ?? [];
        $validated['custom_fields'] = $validated['custom_fields'] ?? [];
        $validated['is_active'] = $request->has('is_active') ? 1 : 0;
        $validated['display_order'] = $validated['display_order'] ?? 10;

        $template->update($validated);

        return redirect()->route('admin.templates.index')
            ->with('success', 'قالب با موفقیت بروزرسانی شد.');
    }

    /**
     * حذف قالب
     */
    public function destroy(CategoryTemplate $template)
    {
        $template->delete();
        return redirect()->route('admin.templates.index')
            ->with('success', 'قالب با موفقیت حذف شد.');
    }

    /**
     * پیش‌نمایش زنده قالب بر اساس داده‌های فرم
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Contracts\View\View
     */
    public function preview(Request $request)
    {
        // اعتبارسنجی داده‌ها (در صورت نیاز می‌توانید قوانین را کامل‌تر کنید)
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'features' => 'array',
            // سایر قوانین اعتبارسنجی در صورت نیاز
        ]);

        // ارسال داده‌ها به ویوی پیش‌نمایش (باید این ویو را ایجاد کنید)
        return view('admin.templates.preview', [
            'template' => (object) $validated
        ]);
    }

    /**
     * جستجوی ایجکس قالب‌ها
     * این متد فقط ردیف‌های جدول را به صورت partial view بازمی‌گرداند.
     */
    public function ajaxSearch(Request $request)
    {
        try {
            $query = $request->get('q', '');
            $templates = CategoryTemplate::with('category')
                ->when($query, function($q) use ($query) {
                    $q->where(function($q) use ($query) {
                        $q->where('template_name', 'like', "%{$query}%")
                          ->orWhereHas('category', function($q2) use ($query) {
                              $q2->where('category_fa', 'like', "%{$query}%");
                          });
                    });
                })
                ->orderBy('template_name')
                ->get();

            if ($templates->isEmpty()) {
                return response()->json([
                    'html' => '<tr><td colspan="4" class="text-center text-muted">نتیجه‌ای یافت نشد.</td></tr>'
                ]);
            }

            $html = view('admin.templates._table_rows', compact('templates'))->render();
            return response()->json(['html' => $html]);
        } catch (\Exception $e) {
            \Log::error('خطا در جستجوی قالب‌ها: ' . $e->getMessage());
            return response()->json([
                'error' => 'خطا در جستجو. لطفا دوباره تلاش کنید.'
            ], 500);
        }
    }
} 