<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\City;
use App\Models\Country;
use App\Models\County;
use App\Models\District;
use App\Models\Province;
use App\Models\Village;
use App\Models\Zone;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class GeographyController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $routeName = $request->route()->getName();

        if (str_contains($routeName, 'countries')) {
            $items = Country::paginate(15);

            return view('admin.geography.countries.index', compact('items'));
        } elseif (str_contains($routeName, 'provinces')) {
            $items = Province::with('country')->paginate(15);

            return view('admin.geography.provinces.index', compact('items'));
        } elseif (str_contains($routeName, 'counties')) {
            $items = County::with('province')->paginate(15);

            return view('admin.geography.counties.index', compact('items'));
        } elseif (str_contains($routeName, 'districts')) {
            $districts = District::with('county')->paginate(15);

            return view('admin.geography.districts.index', compact('districts'));
        } elseif (str_contains($routeName, 'cities')) {
            $items = City::with(['district', 'county'])->paginate(15);

            return view('admin.geography.cities.index', compact('items'));
        } elseif (str_contains($routeName, 'villages')) {
            $villages = Village::with(['district', 'county', 'city'])->paginate(15);

            return view('admin.geography.villages.index', compact('villages'));
        } elseif (str_contains($routeName, 'zones')) {
            $zones = Zone::with(['city', 'village', 'parentZone'])->paginate(15);

            return view('admin.geography.zones.index', compact('zones'));
        }

        // Default fallback
        return redirect()->route('admin.geography.countries.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $routeName = $request->route()->getName();

        if (str_contains($routeName, 'countries')) {
            return view('admin.geography.countries.create');
        } elseif (str_contains($routeName, 'provinces')) {
            $countries = Country::pluck('name', 'id');

            return view('admin.geography.provinces.create', compact('countries'));
        } elseif (str_contains($routeName, 'counties')) {
            $provinces = Province::pluck('name', 'id');

            return view('admin.geography.counties.create', compact('provinces'));
        } elseif (str_contains($routeName, 'districts')) {
            $counties = County::pluck('name', 'id');

            return view('admin.geography.districts.create', compact('counties'));
        } elseif (str_contains($routeName, 'cities')) {
            $districts = District::pluck('name', 'id');
            $counties = County::pluck('name', 'id');

            return view('admin.geography.cities.create', compact('districts', 'counties'));
        } elseif (str_contains($routeName, 'villages')) {
            $districts = District::pluck('name', 'id');
            $counties = County::pluck('name', 'id');
            $cities = City::pluck('name', 'id');

            return view('admin.geography.villages.create', compact('districts', 'counties', 'cities'));
        } elseif (str_contains($routeName, 'zones')) {
            $cities = City::pluck('name', 'id');
            $villages = Village::pluck('name', 'id');
            $parentZones = Zone::pluck('name', 'id');

            return view('admin.geography.zones.create', compact('cities', 'villages', 'parentZones'));
        }

        // Default fallback
        return redirect()->route('admin.geography.countries.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $routeName = $request->route()->getName();

        if (str_contains($routeName, 'countries')) {
            $validated = $request->validate([
                'name' => 'required|string|max:100',
                'name_en' => 'required|string|max:100',
                'iso_code' => 'required|string|size:2|unique:state1_countries,iso_code',
                'iso_code3' => 'required|string|size:3|unique:state1_countries,iso_code3',
                'phone_code' => 'nullable|string|max:10',
                'is_active' => 'boolean',
            ]);

            $country = Country::create($validated);

            return redirect()->route('admin.geography.countries.index')
                ->with('success', 'کشور با موفقیت ایجاد شد');

        } elseif (str_contains($routeName, 'provinces')) {
            $validated = $request->validate([
                'country_id' => 'required|exists:state1_countries,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state2_provinces,slug',
                'center_city_name' => 'nullable|string|max:100',
                'code' => 'nullable|string|max:10',
                'boundaries' => 'nullable',
                'is_active' => 'boolean',
            ]);

            // Handle GeoJSON boundaries for GEOMETRY column
            if (! empty($validated['boundaries'])) {
                try {
                    // Parse the GeoJSON to ensure it's valid
                    $geoJson = json_decode($validated['boundaries'], true);

                    if (json_last_error() === JSON_ERROR_NONE) {
                        // Valid JSON - we need to convert it to WKT format for MySQL GEOMETRY column

                        // Find the polygon coordinates
                        $coordinates = null;

                        // Check for different GeoJSON formats and extract coordinates
                        if (isset($geoJson['type']) && $geoJson['type'] === 'Polygon' && isset($geoJson['coordinates'])) {
                            // Simple Polygon
                            $coordinates = $geoJson['coordinates'][0]; // First ring of coordinates
                        } elseif (isset($geoJson['type']) && $geoJson['type'] === 'FeatureCollection' &&
                                isset($geoJson['features']) && count($geoJson['features']) > 0) {
                            // FeatureCollection
                            foreach ($geoJson['features'] as $feature) {
                                if (isset($feature['geometry']) && isset($feature['geometry']['type']) &&
                                    $feature['geometry']['type'] === 'Polygon') {
                                    $coordinates = $feature['geometry']['coordinates'][0];
                                    break; // Use the first polygon found
                                }
                            }
                        } elseif (isset($geoJson['type']) && $geoJson['type'] === 'Feature' &&
                                isset($geoJson['geometry']) && isset($geoJson['geometry']['type']) &&
                                $geoJson['geometry']['type'] === 'Polygon') {
                            // Feature with Polygon
                            $coordinates = $geoJson['geometry']['coordinates'][0];
                        }

                        if ($coordinates) {
                            // Convert coordinates to WKT format
                            $wktPoints = [];
                            foreach ($coordinates as $coord) {
                                // Ensure we have valid coordinates
                                if (count($coord) >= 2 && is_numeric($coord[0]) && is_numeric($coord[1])) {
                                    $wktPoints[] = $coord[0].' '.$coord[1];
                                }
                            }

                            if (count($wktPoints) >= 3) { // Polygon needs at least 3 points
                                // Create the WKT string and remove from $validated
                                $wkt = 'POLYGON(('.implode(',', $wktPoints).'))';

                                // Remove boundaries from validated data
                                unset($validated['boundaries']);

                                // Create the province first to get its ID
                                $province = Province::create($validated);

                                // Then update the boundaries field directly with SQL
                                DB::statement('
                                    UPDATE state2_provinces 
                                    SET boundaries = ST_GeomFromText(?)
                                    WHERE id = ?
                                ', [$wkt, $province->id]);

                                return redirect()->route('admin.geography.provinces.index')
                                    ->with('success', 'استان با موفقیت ایجاد شد');
                            } else {
                                return redirect()->back()
                                    ->with('error', 'Polygon must have at least 3 points')
                                    ->withInput();
                            }
                        } else {
                            return redirect()->back()
                                ->with('error', 'Could not find polygon coordinates in GeoJSON')
                                ->withInput();
                        }
                    } else {
                        // Invalid JSON
                        unset($validated['boundaries']);

                        return redirect()->back()
                            ->with('error', 'Invalid GeoJSON format for boundaries')
                            ->withInput();
                    }
                } catch (\Exception $e) {
                    // Handle exceptions
                    unset($validated['boundaries']);

                    return redirect()->back()
                        ->with('error', 'Error processing GeoJSON: '.$e->getMessage())
                        ->withInput();
                }
            }

            // If we get here, there were no boundaries
            $province = Province::create($validated);

            return redirect()->route('admin.geography.provinces.index')
                ->with('success', 'استان با موفقیت ایجاد شد');

        } elseif (str_contains($routeName, 'counties')) {
            $validated = $request->validate([
                'province_id' => 'required|exists:state2_provinces,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state3_counties,slug',
                'center_city_name' => 'nullable|string|max:100',
                'code' => 'nullable|string|max:10',
                'is_active' => 'boolean',
            ]);

            $county = County::create($validated);

            return redirect()->route('admin.geography.counties.index')
                ->with('success', 'شهرستان با موفقیت ایجاد شد');

        } elseif (str_contains($routeName, 'districts')) {
            $validated = $request->validate([
                'county_id' => 'required|exists:state3_counties,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state4_districts,slug',
                'center_city_name' => 'nullable|string|max:100',
                'code' => 'nullable|string|max:10',
                'is_active' => 'boolean',
            ]);

            $district = District::create($validated);

            return redirect()->route('admin.geography.districts.index')
                ->with('success', 'بخش با موفقیت ایجاد شد');

        } elseif (str_contains($routeName, 'cities')) {
            $validated = $request->validate([
                'district_id' => 'nullable|exists:state4_districts,id',
                'county_id' => 'required|exists:state3_counties,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state5_cities,slug',
                'code' => 'nullable|string|max:10',
                'is_capital' => 'boolean',
                'is_active' => 'boolean',
            ]);

            $city = City::create($validated);

            return redirect()->route('admin.geography.cities.index')
                ->with('success', 'شهر با موفقیت ایجاد شد');

        } elseif (str_contains($routeName, 'villages')) {
            $validated = $request->validate([
                'district_id' => 'nullable|exists:state4_districts,id',
                'county_id' => 'required|exists:state3_counties,id',
                'city_id' => 'nullable|exists:state5_cities,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state6_village,slug',
                'code' => 'nullable|string|max:10',
                'is_active' => 'boolean',
            ]);

            $village = Village::create($validated);

            return redirect()->route('admin.geography.villages.index')
                ->with('success', 'روستا با موفقیت ایجاد شد');

        } elseif (str_contains($routeName, 'zones')) {
            $validated = $request->validate([
                'city_id' => 'nullable|exists:state5_cities,id',
                'village_id' => 'nullable|exists:state6_village,id',
                'parent_zone_id' => 'nullable|exists:state7_zone,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state7_zone,slug',
                'zone_type' => 'required|string|in:NEIGHBORHOOD,TOWNSHIP,DISTRICT,RESIDENTIAL_COMPLEX,INDUSTRIAL_ZONE,COMMERCIAL_ZONE,OTHER',
                'is_active' => 'boolean',
            ]);

            // Validate that at least one of city_id or village_id is provided
            if (empty($validated['city_id']) && empty($validated['village_id'])) {
                return redirect()->back()
                    ->with('error', 'یک منطقه باید به یک شهر یا روستا متصل باشد')
                    ->withInput();
            }

            $zone = Zone::create($validated);

            return redirect()->route('admin.geography.zones.index')
                ->with('success', 'منطقه با موفقیت ایجاد شد');
        }

        return redirect()->back()->with('error', 'عملیات ناموفق بود');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id, Request $request)
    {
        $routeName = $request->route()->getName();

        if (str_contains($routeName, 'countries')) {
            $item = Country::findOrFail($id);
            $provinces = Province::where('country_id', $id)->get();

            return view('admin.geography.countries.show', compact('item', 'provinces'));
        } elseif (str_contains($routeName, 'provinces')) {
            $item = Province::with('country')->findOrFail($id);
            $counties = County::where('province_id', $id)->get();

            return view('admin.geography.provinces.show', compact('item', 'counties'));
        } elseif (str_contains($routeName, 'counties')) {
            $item = County::with('province')->findOrFail($id);
            $districts = District::where('county_id', $id)->get();

            return view('admin.geography.counties.show', compact('item', 'districts'));
        } elseif (str_contains($routeName, 'districts')) {
            $item = District::with('county')->findOrFail($id);
            $cities = City::where('district_id', $id)->get();
            $villages = Village::where('district_id', $id)->get();

            return view('admin.geography.districts.show', compact('item', 'cities', 'villages'));
        } elseif (str_contains($routeName, 'cities')) {
            $item = City::with(['district', 'county'])->findOrFail($id);
            $zones = Zone::where('city_id', $id)->get();

            return view('admin.geography.cities.show', compact('item', 'zones'));
        } elseif (str_contains($routeName, 'villages')) {
            $item = Village::with(['district', 'county', 'city'])->findOrFail($id);
            $zones = Zone::where('village_id', $id)->get();

            return view('admin.geography.villages.show', compact('item', 'zones'));
        } elseif (str_contains($routeName, 'zones')) {
            $item = Zone::with(['city', 'village', 'parentZone'])->findOrFail($id);
            $childZones = Zone::where('parent_zone_id', $id)->get();

            return view('admin.geography.zones.show', compact('item', 'childZones'));
        }

        return redirect()->back()->with('error', 'مورد یافت نشد');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id, Request $request)
    {
        $routeName = $request->route()->getName();

        if (str_contains($routeName, 'countries')) {
            $item = Country::findOrFail($id);

            return view('admin.geography.countries.edit', compact('item'));
        } elseif (str_contains($routeName, 'provinces')) {
            $item = Province::findOrFail($id);
            $countries = Country::pluck('name', 'id');

            return view('admin.geography.provinces.edit', compact('item', 'countries'));
        } elseif (str_contains($routeName, 'counties')) {
            $item = County::findOrFail($id);
            $provinces = Province::pluck('name', 'id');

            return view('admin.geography.counties.edit', compact('item', 'provinces'));
        } elseif (str_contains($routeName, 'districts')) {
            $item = District::findOrFail($id);
            $counties = County::pluck('name', 'id');

            return view('admin.geography.districts.edit', compact('item', 'counties'));
        } elseif (str_contains($routeName, 'cities')) {
            $item = City::findOrFail($id);
            $districts = District::pluck('name', 'id');
            $counties = County::pluck('name', 'id');

            return view('admin.geography.cities.edit', compact('item', 'districts', 'counties'));
        } elseif (str_contains($routeName, 'villages')) {
            $item = Village::findOrFail($id);
            $districts = District::pluck('name', 'id');
            $counties = County::pluck('name', 'id');
            $cities = City::pluck('name', 'id');

            return view('admin.geography.villages.edit', compact('item', 'districts', 'counties', 'cities'));
        } elseif (str_contains($routeName, 'zones')) {
            $item = Zone::findOrFail($id);
            $cities = City::pluck('name', 'id');
            $villages = Village::pluck('name', 'id');
            $parentZones = Zone::where('id', '!=', $id)->pluck('name', 'id');

            return view('admin.geography.zones.edit', compact('item', 'cities', 'villages', 'parentZones'));
        }

        return redirect()->back()->with('error', 'مورد یافت نشد');
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $routeName = $request->route()->getName();

        if (str_contains($routeName, 'countries')) {
            $validated = $request->validate([
                'name' => 'required|string|max:100',
                'name_en' => 'required|string|max:100',
                'iso_code' => 'required|string|size:2|unique:state1_countries,iso_code,'.$id,
                'iso_code3' => 'required|string|size:3|unique:state1_countries,iso_code3,'.$id,
                'phone_code' => 'nullable|string|max:10',
                'is_active' => 'boolean',
            ]);

            $country = Country::findOrFail($id);
            $country->update($validated);

            return redirect()->route('admin.geography.countries.index')
                ->with('success', 'کشور با موفقیت بروزرسانی شد');
        } elseif (str_contains($routeName, 'provinces')) {
            $validated = $request->validate([
                'country_id' => 'required|exists:state1_countries,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state2_provinces,slug,'.$id,
                'center_city_name' => 'nullable|string|max:100',
                'code' => 'nullable|string|max:10',
                'boundaries' => 'nullable',
                'is_active' => 'boolean',
            ]);

            // Handle GeoJSON boundaries for GEOMETRY column
            if (! empty($validated['boundaries'])) {
                try {
                    // Parse the GeoJSON to ensure it's valid
                    $geoJson = json_decode($validated['boundaries'], true);

                    if (json_last_error() === JSON_ERROR_NONE) {
                        // Valid JSON - we need to convert it to WKT format for MySQL GEOMETRY column

                        // Find the polygon coordinates
                        $coordinates = null;

                        // Check for different GeoJSON formats and extract coordinates
                        if (isset($geoJson['type']) && $geoJson['type'] === 'Polygon' && isset($geoJson['coordinates'])) {
                            // Simple Polygon
                            $coordinates = $geoJson['coordinates'][0]; // First ring of coordinates
                        } elseif (isset($geoJson['type']) && $geoJson['type'] === 'FeatureCollection' &&
                                isset($geoJson['features']) && count($geoJson['features']) > 0) {
                            // FeatureCollection
                            foreach ($geoJson['features'] as $feature) {
                                if (isset($feature['geometry']) && isset($feature['geometry']['type']) &&
                                    $feature['geometry']['type'] === 'Polygon') {
                                    $coordinates = $feature['geometry']['coordinates'][0];
                                    break; // Use the first polygon found
                                }
                            }
                        } elseif (isset($geoJson['type']) && $geoJson['type'] === 'Feature' &&
                                isset($geoJson['geometry']) && isset($geoJson['geometry']['type']) &&
                                $geoJson['geometry']['type'] === 'Polygon') {
                            // Feature with Polygon
                            $coordinates = $geoJson['geometry']['coordinates'][0];
                        }

                        if ($coordinates) {
                            // Convert coordinates to WKT format
                            $wktPoints = [];
                            foreach ($coordinates as $coord) {
                                // Ensure we have valid coordinates
                                if (count($coord) >= 2 && is_numeric($coord[0]) && is_numeric($coord[1])) {
                                    $wktPoints[] = $coord[0].' '.$coord[1];
                                }
                            }

                            if (count($wktPoints) >= 3) { // Polygon needs at least 3 points
                                // Create the WKT string and don't store in $validated
                                // We'll handle the update outside the model's mass assignment
                                $wkt = 'POLYGON(('.implode(',', $wktPoints).'))';

                                // Remove boundaries from validated data
                                unset($validated['boundaries']);

                                // Find the province to update other fields first
                                $province = Province::findOrFail($id);
                                $province->update($validated);

                                // Then update the boundaries field directly with SQL
                                DB::statement('
                                    UPDATE state2_provinces 
                                    SET boundaries = ST_GeomFromText(?)
                                    WHERE id = ?
                                ', [$wkt, $id]);

                                return redirect()->route('admin.geography.provinces.index')
                                    ->with('success', 'استان با موفقیت بروزرسانی شد');
                            } else {
                                return redirect()->back()
                                    ->with('error', 'Polygon must have at least 3 points')
                                    ->withInput();
                            }
                        } else {
                            return redirect()->back()
                                ->with('error', 'Could not find polygon coordinates in GeoJSON')
                                ->withInput();
                        }
                    } else {
                        // Invalid JSON
                        unset($validated['boundaries']);

                        return redirect()->back()
                            ->with('error', 'Invalid GeoJSON format for boundaries')
                            ->withInput();
                    }
                } catch (\Exception $e) {
                    // Handle exceptions
                    unset($validated['boundaries']);

                    return redirect()->back()
                        ->with('error', 'Error processing GeoJSON: '.$e->getMessage())
                        ->withInput();
                }
            }

            // If we get here, there were no boundaries or we need to update without boundaries
            $province = Province::findOrFail($id);
            $province->update($validated);

            return redirect()->route('admin.geography.provinces.index')
                ->with('success', 'استان با موفقیت بروزرسانی شد');
        } elseif (str_contains($routeName, 'counties')) {
            $validated = $request->validate([
                'province_id' => 'required|exists:state2_provinces,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state3_counties,slug,'.$id,
                'center_city_name' => 'nullable|string|max:100',
                'code' => 'nullable|string|max:10',
                'is_active' => 'boolean',
            ]);

            $county = County::findOrFail($id);
            $county->update($validated);

            return redirect()->route('admin.geography.counties.index')
                ->with('success', 'شهرستان با موفقیت بروزرسانی شد');
        } elseif (str_contains($routeName, 'districts')) {
            $validated = $request->validate([
                'county_id' => 'required|exists:state3_counties,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state4_districts,slug,'.$id,
                'center_city_name' => 'nullable|string|max:100',
                'code' => 'nullable|string|max:10',
                'is_active' => 'boolean',
            ]);

            $district = District::findOrFail($id);
            $district->update($validated);

            return redirect()->route('admin.geography.districts.index')
                ->with('success', 'بخش با موفقیت بروزرسانی شد');
        } elseif (str_contains($routeName, 'cities')) {
            $validated = $request->validate([
                'district_id' => 'nullable|exists:state4_districts,id',
                'county_id' => 'required|exists:state3_counties,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state5_cities,slug,'.$id,
                'code' => 'nullable|string|max:10',
                'is_capital' => 'boolean',
                'is_active' => 'boolean',
            ]);

            $city = City::findOrFail($id);
            $city->update($validated);

            return redirect()->route('admin.geography.cities.index')
                ->with('success', 'شهر با موفقیت بروزرسانی شد');
        } elseif (str_contains($routeName, 'villages')) {
            $validated = $request->validate([
                'district_id' => 'nullable|exists:state4_districts,id',
                'county_id' => 'required|exists:state3_counties,id',
                'city_id' => 'nullable|exists:state5_cities,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state6_village,slug,'.$id,
                'code' => 'nullable|string|max:10',
                'is_active' => 'boolean',
            ]);

            $village = Village::findOrFail($id);
            $village->update($validated);

            return redirect()->route('admin.geography.villages.index')
                ->with('success', 'روستا با موفقیت بروزرسانی شد');
        } elseif (str_contains($routeName, 'zones')) {
            $validated = $request->validate([
                'city_id' => 'nullable|exists:state5_cities,id',
                'village_id' => 'nullable|exists:state6_village,id',
                'parent_zone_id' => 'nullable|exists:state7_zone,id',
                'name' => 'required|string|max:100',
                'name_en' => 'nullable|string|max:100',
                'slug' => 'required|string|max:100|unique:state7_zone,slug,'.$id,
                'zone_type' => 'required|string|in:NEIGHBORHOOD,TOWNSHIP,DISTRICT,RESIDENTIAL_COMPLEX,INDUSTRIAL_ZONE,COMMERCIAL_ZONE,OTHER',
                'is_active' => 'boolean',
            ]);

            // Ensure we're not setting a zone as its own parent
            if (! empty($validated['parent_zone_id']) && $validated['parent_zone_id'] == $id) {
                return redirect()->back()
                    ->with('error', 'یک منطقه نمی‌تواند به عنوان والد خودش تنظیم شود')
                    ->withInput();
            }

            $zone = Zone::findOrFail($id);
            $zone->update($validated);

            return redirect()->route('admin.geography.zones.index')
                ->with('success', 'منطقه با موفقیت بروزرسانی شد');
        }

        return redirect()->back()->with('error', 'عملیات ناموفق بود');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id, Request $request)
    {
        $routeName = $request->route()->getName();

        try {
            if (str_contains($routeName, 'countries')) {
                $country = Country::findOrFail($id);

                // Check if country has provinces
                if ($country->provinces()->count() > 0) {
                    return response()->json([
                        'success' => false,
                        'message' => 'این کشور دارای استان‌هایی است و نمی‌توان آن را حذف کرد',
                    ]);
                }

                $country->delete();

                return response()->json([
                    'success' => true,
                    'message' => 'کشور با موفقیت حذف شد',
                ]);
            }

            // Similar handling for other geographical entities

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در حذف: '.$e->getMessage(),
            ], 500);
        }

        return response()->json([
            'success' => false,
            'message' => 'عملیات ناموفق بود',
        ], 400);
    }

    /**
     * Get provinces by country for dropdowns
     */
    public function getProvincesByCountry($countryId)
    {
        $provinces = Province::where('country_id', $countryId)
            ->where('is_active', 1)
            ->orderBy('name')
            ->get(['id', 'name']);

        return response()->json($provinces);
    }

    /**
     * Get counties by province for dropdowns
     */
    public function getCountiesByProvince($provinceId)
    {
        $counties = County::where('province_id', $provinceId)
            ->where('is_active', 1)
            ->orderBy('name')
            ->get(['id', 'name']);

        return response()->json($counties);
    }

    /**
     * Get districts by county for dropdowns
     */
    public function getDistrictsByCounty($countyId)
    {
        $districts = District::where('county_id', $countyId)
            ->where('is_active', 1)
            ->orderBy('name')
            ->get(['id', 'name']);

        return response()->json($districts);
    }

    /**
     * Get cities by district for dropdowns
     */
    public function getCitiesByDistrict($districtId)
    {
        $cities = City::where('district_id', $districtId)
            ->where('is_active', 1)
            ->orderBy('name')
            ->get(['id', 'name']);

        return response()->json($cities);
    }

    /**
     * Get villages by district for dropdowns
     */
    public function getVillagesByDistrict($districtId)
    {
        $villages = Village::where('district_id', $districtId)
            ->where('is_active', 1)
            ->orderBy('name')
            ->get(['id', 'name']);

        return response()->json($villages);
    }

    /**
     * Attempt to fix corrupted GeoJSON data for a province
     */
    public function fixBoundaries(string $id)
    {
        try {
            $province = Province::findOrFail($id);

            // Use MySQL's ST_GeomFromText function to properly set a GEOMETRY column
            // This is the correct way to store WKT in a GEOMETRY field
            DB::statement("
                UPDATE state2_provinces 
                SET boundaries = ST_GeomFromText('POLYGON((51.3 35.7, 51.5 35.7, 51.5 35.8, 51.3 35.8, 51.3 35.7))')
                WHERE id = ?
            ", [$id]);

            return redirect()->route('admin.geography.provinces.show', $id)
                ->with('success', 'The boundaries data has been replaced with a default polygon for Tehran.');

        } catch (\Exception $e) {
            return redirect()->route('admin.geography.provinces.index')
                ->with('error', 'Error fixing boundaries: '.$e->getMessage());
        }
    }
}
